/****************************************************************************
 sr.h

 (C) Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.6  $
   $Author:   mlentczn  $
     $Date:   24 Feb 1992 11:07:34  $

 clsSR inherits from clsObject.

 clsSR is the class of theSearchManager.  It defines a protocol which
 clients can respond to implement Find and Replace.  Clients of this 
 protocol must respond to the "mark" protocol defined in mark.h.
****************************************************************************/

/****  Debugging Flags  ****/
/*
 The Find and Replace mechanism uses the debug flag R10000.
*/

#ifndef SR_INCLUDED
#define SR_INCLUDED 1

#ifndef MARK_INCLUDED
#include <mark.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define srBufSize		80

typedef struct SR_FLAGS {
	BOOLEAN		matchCase		: 1, 	// case must match
				matchWord		: 1,	// full word search
				keepOldCase		: 1,	// replace with found case
				findFromEdge	: 1,	// search from edge of doc
				onBigCard		: 1;	// display big card
} SR_FLAGS;


typedef struct SR_METRICS {
    CHAR				findText[srBufSize];
    CHAR				replaceText[srBufSize];
	MARK_MSG_FLAGS		markFlags;
    SR_FLAGS			searchFlags;
} SR_METRICS, * P_SR_METRICS;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Statuses									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* The current match cannot/may not be replaced. */
#define stsSRCannotReplace      MakeStatus(clsSR, 1)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				Messages Sent to Clients via clsMark					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgSRNextChars					takes P_SR_NEXT_CHARS, returns STATUS
	Asks the client to move the token to the next group of characters.

 Important:  your handler must have the following as its first statement.
 Replace "clsYourClassHere" with the uid of your class.  See mark.h.
//{
	MarkHandlerForClass(clsYourClassHere);
//}

 This group may be up to maxLen characters in size.  The client sets the
 len parameter to the actual size of the group, and if the group is the
 start and/or end of a block of character, sets the respective flags.  A
 block is defined as a logically contiguous group of characters that can be	
 searched.

 Any non-text element usually delimits the end of a block.  If the element
 is an embedded window that should be searched, the token should be set to
 point to the embedded window and stsMarkEnterChild (see mark.h) should be
 returned.  If the element is not a child, then it should be simply skipped
 and the token moved to the next group of characters following it.

 Example:  If the following text is in the client's data, and 
 msgSRNextChars is received with a maxLen of 5, the token would should
 refer to the blocks 1 through 4 in succession.  blockStart should be true
 for blocks 1 and 3 and blockEnd should be true for blocks 2 and 4.  In
 this way, "SEN" and "MANTLE" can be found, but "GERMAN" which spans
 some non-text object won't be mistakenly found.
//{
		M E S S E N G E R (non-text-thing) M A N T L E
	   |         |       |                |         | |
	   +----1----+---2---+----------------+----3----+4+
//}
*/
#define msgSRNextChars				MakeMsg(clsSR, 1)

typedef struct SR_NEXT_CHARS {
	MARK_MSG_HEADER	header;
	U32				maxLen;			// In: maximum size the group can be
	U32				len;			// Out: the size of the group
	BOOLEAN			blockStart;		// Out: the group starts a block
	BOOLEAN			blockEnd;		// Out: the group ends a block
} SR_NEXT_CHARS, * P_SR_NEXT_CHARS;


/****************************************************************************
 msgSRGetChars					takes P_SR_GET_CHARS, returns STATUS
	The component passes back the characters from the location
	identified by the token.
	
 Important:  your handler must have the following as its first statement.
 Replace "clsYourClassHere" with the uid of your class.  See mark.h.
//{
	MarkHandlerForClass(clsYourClassHere);
//}

 pArgs->first is token-relative and pArgs->len is the number of characters
 to return.  Thus (0,2) requests the first two characters, (1,1) requests
 the second character, and (3,0) requests no characters.

 The string returned must be null-terminated.  Note that if len is less
 than bufLen then this is always possible without truncation.  Otherwise,
 the number of characters returned should be one less than bufLen and they
 should still be null terminated.
*/
#define msgSRGetChars				MakeMsg(clsSR, 2)

typedef struct SR_GET_CHARS {
	MARK_MSG_HEADER	header;
	U32				first;			// In: character to start with
	U32				len;			// In: the number of characters to return
	U32				bufLen;			// In: length of the buffer
	P_CHAR			pBuf;			// In: pointer to the buffer to fill
} SR_GET_CHARS, * P_SR_GET_CHARS;


/****************************************************************************
 msgSRReplaceChars			takes P_SR_REPLACE_CHARS, returns STATUS
	Ask the component to replace some of the characters at the location 
	identified by the token.
	
 Important:  your handler must have the following as its first statement.
 Replace "clsYourClassHere" with the uid of your class.  See mark.h.
//{
	MarkHandlerForClass(clsYourClassHere);
//}

 pArgs->first is token-relative, and pArgs->len is the number of characters
 to replace.  Thus (0,2) replaces the first two characters, (1,1) replaces
 the second character, and (3,0) replaces no characters starting between
 the third and fourth (thus effecting an insertion).
 
 pArgs->first may be negative, indicating replacement of text BEFORE the
 current token (or large indicating AFTER).  However, in no case will
 pArgs->first go beyond the boundaries indicated by the blockStart and
 blockEnd flags from previous calls to msgSRNextChars.
 
 This message should only affect the token insofar as the replacement makes
 changes to the data the token refers to.  For example: if the token refers
 to the three characters "cat" and the replace messages changes the
 substring "c" (0,1) into "womb", then the token should now refer to the
 six characters "wombat".
*/								 
#define msgSRReplaceChars			MakeMsg(clsSR, 3)

typedef struct SR_REPLACE_CHARS {
	MARK_MSG_HEADER	header;
	S32				first;			// In: replacement starts here
	U32				len;			// In: ...and is this long
	U32				bufLen;			// In: repl. size in characters
	P_CHAR			pBuf;			// In: the buffer of the characters
} SR_REPLACE_CHARS, * P_SR_REPLACE_CHARS;


/****************************************************************************
 msgSRPositionChars			takes P_SR_POSITION_CHARS, returns STATUS
	Asks the component to reposition the token to some of the characters 
	in the current group.
	
 Important:  your handler must have the following as its first statement.
 Replace "clsYourClassHere" with the uid of your class.  See mark.h.
//{
	MarkHandlerForClass(clsYourClassHere);
//}

 pArgs->first is token-relative, and pArgs->len is the number of characters
 to reposition to.  Thus (0,2) positions to the first two characters, (1,1)
 positions to the second character, and (3,0) positions to between the
 third and fourth characters.
 
 pArgs->first may be negative indicating positioning BEFORE the current
 token (or large indicating AFTER). However, in no case will pArgs->first
 go beyond the boundaries indicated by the blockStart and blockEnd flags
 from previous calls to msgSRNextChars.
*/
#define msgSRPositionChars			MakeMsg(clsSR, 4)
			 
typedef struct SR_POSITION_CHARS {
	MARK_MSG_HEADER	header;
	S32				first;			// In: new position starts here
	U32				len;			// In: ...and is this long
} SR_POSITION_CHARS, * P_SR_POSITION_CHARS;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Messages to theSearchManager					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* 
 * These messages are sent to theSearchManager by PenPoint's standard UI
 * elements.  Typical clients do not send them.
*/

/****************************************************************************
 msgSRInvokeSearch   takes P_SR_INVOKE_SEARCH, returns STATUS
 	Starts a Find & Replace option sheet.
							
 The target of the search is the target argument.  However if fromSelection
 is true then it is the selection; or if fromGesture is true then it is
 from the gesture.				   

 The user's last saved metrics are always used except that
	-:	metrics.findText is used if it is not the empty string
	-:	metrics.replaceText	is used if it is not the empty string
    -:  metrics.markFlags & metrics.searchFlags are used if pArgs->useFlags
        is true

 If doFind is true, then an initial find is executed.

 If noUI is true, then the option sheet isn't created. This is only useful
 in conjunction with doFind (otherwise, nothing has happened!), the result
 being a "find next" operation.

 If useWord is true, then the find text will be fetched from the target
 with msgSRGetChars.
*/
#define msgSRInvokeSearch  		    MakeMsg(clsSR, 10)

typedef struct SR_INVOKE_SEARCH {
	OBJECT			target;				// nil if fromGesture or fromSelection
    BOOLEAN			fromSelection	:1,	// start from the selection
					fromGesture		:1, // start from the gesture given
					doFind  		:1,	// do an initial find
					findBackward	:1,	// direction for initial find
					noUI    		:1,	// don't open option sheet
					useWord 		:1,	// use the word at the gesture or selection
					useFlags		:1;	// use the flags in metrics
	U16				reserved;
	GWIN_GESTURE	gesture;			// the gesture if fromGesture
	SR_METRICS		metrics;			// optional initial text and flags
	U32				reserved2;
} SR_INVOKE_SEARCH, * P_SR_INVOKE_SEARCH;


/****************************************************************************
 msgSRRememberMetrics		takes P_SR_METRICS, returns STATUS
	Asks theSearchManager to remember the current settings of a Find &
	Replace option sheet
 			
 As a result, when theSearchManager option sheet next appears it will have
 the these settings.														  
*/	
#define msgSRRememberMetrics		MakeMsg(clsSR, 12)

#endif
