/****************************************************************************
 File: progress.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.20  $
   $Author:   cmeyer  $
     $Date:   18 Mar 1992 16:58:42  $

 This file contains the API for clsProgressBar.

 clsProgressBar inherits from clsControl.
 Implements a read-only or read/write progress indicator.
****************************************************************************/


/**** Debugging Flags ****/
/*
 The clsProgressBar debugging flag is 'K'.  Defined values are:

    flag14 (0x4000):    general
*/

#ifndef PROGRESS_INCLUDED
#define PROGRESS_INCLUDED

												#ifndef CONTROL_INCLUDED
#include <control.h>
												#endif
												#ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>
												#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Common #defines and typedefs					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

// Labels style
#define psLabelsNumeric			0
#define psLabelsNone			1
#define psLabelsCustom			2

// Ticks style
#define psTicksSmall			0
#define psTicksFull				1
#define psTicksNone				2

// Direction style
#define	psDirectionHorizontal	0	// horizontal indicator
#define	psDirectionVertical		1	// vertical indicator

// Thickness style
#define psThicknessRelFont		0	// thickness varies with system font size
#define psThicknessFixed		1	// thickness is fixed

// Edge Styles
#define	psEdgeNone					0
#define	psEdgeMinLat			    flag0
#define	psEdgeMaxLat				flag1
#define	psEdgeMinLong	            flag2
#define	psEdgeMaxLong   			flag3

#define	psEdgeAll					(psEdgeMinLat | psEdgeMaxLat | \
								 	psEdgeMinLong | psEdgeMaxLong)

/*
 'Lat' is latitude, and 'Long' is longitude.  For horizontal progress bars,
 latitude is the y dimension (or minor axis), and longitude is the x
 dimension (or major axis).  For vertical bars, lat is x, and long is y.
*/


typedef struct PROGRESS_STYLE {
	U16	labels			: 2,	// labels style
		ticks			: 2,	// style of ticks to paint
		direction		: 2,	// direction of major axis
		units			: 6,	// units for everything except labels
		thickness		: 2,	// thickness style for lines and ticks
		labelRotation	: 2;	// use lsRotate* from label.h

	U16	labelScaleUnits	: 6,	// scale units for labels from border.h
        edge            : 4,    // bar edges to display
                                // (separate from clsBorder edges)
		labelFontType	: 2,	// use lsFont* from label.h
		spare			: 4;	// unused (reserved)

	U16	spare2	        : 16;	// unused (reserved)
} PROGRESS_STYLE, *P_PROGRESS_STYLE;
	
/*
 Default PROGRESS_STYLE:
//{
	labels			= psLabelsNone
	ticks			= psTicksNone
	direction		= psDirectionHorizontal
	units			= bsUnitsPoints
	thickness		= psThicknessRelFont
	labelRotation	= lsRotateNone
	labelFontType	= lsFontSystem
	labelScaleUnits	= bsUnitsLayout
	edge        	= psEdgeMinLat | psEdgeMinLong
//}
*/

typedef struct PROGRESS_REGION {
	U32				rgb;
	SYSDC_PATTERN	pattern;
} PROGRESS_REGION, *P_PROGRESS_REGION;

typedef struct PROGRESS_METRICS {
	PROGRESS_STYLE	style;				// overall style
	S32				numIntervals;
	S32				ticksPerLabel;		// gives period of labels
	S32				minNumericLabel;	// when psLabelsNumeric
	S32				maxNumericLabel;	// when psLabelsNumeric
	U16				thicknessBase;		// thickness (units or multiplier)
	U16				latitude;			// dimension of minor axis (in units)
	U16				longitude;			// dimension of major axis (in units)
	S32				maxValue;			// values are in [0..maxValue]
	S32				value;				// current value
	SYSDC_FONT_SPEC	font; // spec to open if style.labelFontType == lsFontCustom
	U8				labelScale;	// scale for labels as in border.h
	U32				spare1;		// unused (reserved)
	U32				spare2;		// unused (reserved)
} PROGRESS_METRICS, *P_PROGRESS_METRICS;
	
/*
 metrics.latitude and .longitude are used only when the progress bar is
 shrink-wrapped in those dimensions.  When not shrink-wrapped, the progress
 bar expands to fill the available space.
*/

/****************************************************************************
 msgNew		takes P_PROGRESS_NEW, returns STATUS
	category: class message
	Creates a progress indicator.

 The filled region looks are initialized with:
	rgb = SysDcGrayRGB(128)
	pattern = sysDcPatForeground

 The unfilled region looks are initialized with:
	rgb = sysDcRGBTransparent
	pattern = sysDcPatNil
*/

typedef struct PROGRESS_NEW_ONLY {
	PROGRESS_METRICS	metrics;
	P_CHAR				fontName;	// font name from which to derive font.id
	U32					spare1;		// unused (reserved)
	U32					spare2;		// unused (reserved)
} PROGRESS_NEW_ONLY, *P_PROGRESS_NEW_ONLY;

#define	progressNewFields	\
	controlNewFields		\
	PROGRESS_NEW_ONLY		progress;

typedef struct PROGRESS_NEW {
	progressNewFields
} PROGRESS_NEW, *P_PROGRESS_NEW;


/****************************************************************************
 msgNewDefaults				takes P_PROGRESS_NEW, returns STATUS
	category: class message
	Initializes the PROGRESS_NEW structure to default values.

 Zeroes out pArgs->progress and sets:
//{
	pArgs->win.flags.style |= wsShrinkWrapWidth | wsShrinkWrapHeight;

	pArgs->border.style.previewAlter = bsAlterNone;
	pArgs->border.style.selectedAlter = bsAlterNone;

	pArgs->control.style.showDirty = false;

	pArgs->progress.metrics.style.labels = psLabelsNone;
	pArgs->progress.metrics.style.ticks = psTicksNone;
	pArgs->progress.metrics.style.units = bsUnitsPoints;
	pArgs->progress.metrics.style.labelScaleUnits = bsUnitsLayout;
	pArgs->progress.metrics.style.edge = psEdgeAll;
	pArgs->progress.metrics.numIntervals = 10;
	pArgs->progress.metrics.ticksPerLabel = 2;
	pArgs->progress.metrics.minNumericLabel = 0;
	pArgs->progress.metrics.maxNumericLabel = 100;
	pArgs->progress.metrics.thicknessBase = 1;
	pArgs->progress.metrics.latitude = 18;
	pArgs->progress.metrics.longitude = 144;
	pArgs->progress.metrics.maxValue = 100;
	pArgs->progress.metrics.value = 0;
	pArgs->progress.metrics.labelScale = lsScaleMedium;
//}
 Also sets pArgs->progress.metrics.font to the default system font.
*/


/****************************************************************************
 msgProgressGetStyle			takes P_PROGRESS_STYLE, returns STATUS
	Passes back the current style.
*/
#define msgProgressGetStyle		MakeMsg(clsProgressBar, 1)


/****************************************************************************
 msgProgressSetStyle			takes P_PROGRESS_STYLE, returns STATUS
	Sets the style.

 The progress bar will set its layout bit dirty (as in
 msgWinSetLayoutDirty) as necessary.  It will use msgWinDirtyRect in a
 similar fashion.  It is the client's responsibility to send msgWinLayout
 to the progress bar whenever the style changes would affect the layout.
*/
#define msgProgressSetStyle		MakeMsg(clsProgressBar, 2)


/****************************************************************************
 msgProgressGetMetrics			takes P_PROGRESS_METRICS, returns STATUS
	Passes back the current metrics.
*/
#define msgProgressGetMetrics	MakeMsg(clsProgressBar, 3)


/****************************************************************************
 msgProgressSetMetrics			takes P_PROGRESS_METRICS, returns STATUS
	Sets the metrics.

 The progress bar will set its layout bit dirty (as in
 msgWinSetLayoutDirty) as necessary.  It will use msgWinDirtyRect in a
 similar fashion.  It is the client's responsibility to send msgWinLayout
 to the progress bar whenever the changes would affect the layout.
*/
#define msgProgressSetMetrics	MakeMsg(clsProgressBar, 4)


/****************************************************************************
 msgProgressGetFilled			takes P_PROGRESS_REGION, returns STATUS
	Passes back the current filled region looks.
*/
#define msgProgressGetFilled	MakeMsg(clsProgressBar, 5)


/****************************************************************************
 msgProgressSetFilled			takes P_PROGRESS_REGION, returns STATUS
	Sets the current filled region looks.

 The progress bar will self-send msgWinDirtyRect as necessary.
*/
#define msgProgressSetFilled	MakeMsg(clsProgressBar, 6)


/****************************************************************************
 msgProgressGetUnfilled			takes P_PROGRESS_REGION, returns STATUS
	Passes back the current unfilled region looks.
*/
#define msgProgressGetUnfilled	MakeMsg(clsProgressBar, 7)


/****************************************************************************
 msgProgressSetUnfilled			takes P_PROGRESS_REGION, returns STATUS
	Sets the current unfilled region looks.

 The progress bar will self-send msgWinDirtyRect as necessary.
*/
#define msgProgressSetUnfilled	MakeMsg(clsProgressBar, 8)


/****************************************************************************
 msgProgressProvideLabel		takes P_PROGRESS_PROVIDE_LABEL, returns STATUS
	category: client responsibility
	Sent to the client when style.labels == psLabelsCustom.

 The client should copy a string for the indicated position into
 the provided buffer.
*/
#define msgProgressProvideLabel	MakeMsg(clsProgressBar, 9)

typedef struct PROGRESS_PROVIDE_LABEL {
	CONTROL		progressBar;	//  In: requestor
	U16			position;		//  In: position of label (0 at minimum)
	P_CHAR		pString;		// Out: a 256 byte buffer for the string
	U32			spare;			// unused (reserved)
} PROGRESS_PROVIDE_LABEL, *P_PROGRESS_PROVIDE_LABEL;


/****************************************************************************
 msgProgressGetVisInfo			takes P_PROGRESS_VIS_INFO, returns STATUS
	Passes back information about the current state of the visuals.

 All measurements are in LWC (device units).
*/
#define msgProgressGetVisInfo	MakeMsg(clsProgressBar, 10)

typedef struct PROGRESS_VIS_INFO {
    RECT32  filledRect,
            unfilledRect;
	U32		spare1;		// unused (reserved)
	U32		spare2;		// unused (reserved)
} PROGRESS_VIS_INFO, *P_PROGRESS_VIS_INFO;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *				    Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgControlGetValue				takes P_S32, returns STATUS
	Passes back the receiver's value (metrics.value).
*/


/****************************************************************************
 msgControlSetValue				takes S32, returns STATUS
	Sets the receiver's value.

 The progress bar will self-send msgWinDirtyRect as necessary.
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Causes an object to file itself in an object file.

 clsProgressBar responds by filing away all of its state.
*/


/****************************************************************************
 msgRestore	takes P_OBJ_RESTORE, returns STATUS
	Creates and restores an object from an object file.

 clsLabel responds by restoring all of its state.
*/


/****************************************************************************
 msgWinLayoutSelf takes P_WIN_METRICS, returns STATUS
    Tell a window to layout its children.

 clsProgressBar responds by recomputing its layout parameters.

 If the receiver is shrink-wrapping in a dimension, it will use the
 latitude or longitude value as appropriate to determine the interior
 dimension of the progress bar (which does not include the inked edges
 of the bar).  When not shrink-wrapping in a dimension, the corresponding
 latitude or longitude value is ignored.
*/


/****************************************************************************
 msgWinRepaint takes nothing, returns STATUS
    category: descendant responsibility
    Tells a window to repaint itself.

 clsProgressBar responds by painting the edges, bar, ticks, and labels.

 First, the progressBar self-sends msgControlGetValue to get its current
 value and then msgBorderGetForegroundRGB to get the color in which to
 paint the edges, ticks, and labels.  It then paints the edges.

 Next, the progressBar will paint the unfilled portion of the bar if the
 unfilled pattern isn't sysDcPatNil.  The pattern will be painted with
 the specified foreground RGB and a background RGB obtained by self-sending
 msgBorderGetBackgroundRGB.  See msgProgressSetUnfilled.

 The progressBar will then paint the filled portion of the bar if the
 filled pattern isn't sysDcPatNil.  The method is as described above.
 See msgProgressSetFilled.

 While drawing the tick marks, the progressBar will self-send
 msgBorderRGBToInk and use a foreground color that is opposite so that
 the ticks will show up against the filled/unfilled regions.

 Finally, the labels are painted using a foreground RGB obtained by
 self-sending msgBorderGetForegroundRGB.
*/


/****************************************************************************
 msgWinGetBaseline takes P_WIN_METRICS, returns STATUS
    Gets the desired x,y alignment of a window.

 clsProgressBar responds by setting pArgs->bounds.origin to the origin
 of the bar within self.
*/


#endif	// PROGRESS_INCLUDED

