/****************************************************************************
 File: npitem.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.12  $
   $Author:   bschwart  $
     $Date:   28 Feb 1992 09:32:50  $

 This file contains the API definition for clsNPItem.

 clsNPItem inherits from clsObject.

 NPItem is the item class for PenPoint's ink-management or note-taking
 building block.  While instances of clsNPItem are never created, (subclasses
 like clsNPScribbleItem and clsNPTextItem are more interesting), NPItem
 defines a protocol as well as doing much of the work for basic operations.

 To add new item types to the ink building block, create a subclass of
 clsNPItem that implements the messages defined below in the section:
 "Messages that are usually overridden by subclasses."  Once this new item
 is inserted into a clsNPData object it will show up in the clsNotePaper
 view that observes that object.  The new item will then behave like the
 other item in terms of basic operations like move, copy, deletion, style
 changes, etc.

****************************************************************************/
#ifndef NPITEM_INCLUDED
#define NPITEM_INCLUDED

#ifndef GEO_INCLUDED
#include <geo.h>
#endif
#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef BORDER_INCLUDED
#include <border.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Types and Constants                                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define clsNPItem					MakeGlobalWKN(2569,1)

#define stsNPItemNoSplit			MakeWarning(clsNPItem, 0)

/*
 The NPData object handles versioning for NPItem's and their subclasses.
 If the version of the object being restored matches the runtime version,
 nothing special is done.  However, if there is a difference, the version
 number of the filed object is stamped as a U16 property onto the file
 using tagItemVersion as the property's tag.
*/
#define NP_ITEM_VERSION	1
#define tagItemVersion	MakeTag(clsNPItem, 0)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Messages		                                                       *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*	Next up: 44;  Recycle: 3 */

/****************************************************************************
 msgNewDefaults				takes P_NP_ITEM_NEW, returns STATUS
	Initialize pArgs.

 Zeroes out pArgs->npData and sets:
//{
	pArgs->item.penStyle	= penFineBlack;
//}
*/
typedef struct NP_ITEM_NEW_ONLY {

	RECT32		bounds;
	XY16		baseline;
	BOOLEAN		selected;
	U32			penStyle;  // (Pen styles are defined in notepapr.h.)
	S32			spare2;

} NP_ITEM_NEW_ONLY,  *P_NP_ITEM_NEW_ONLY;

#define	npItemNewFields		\
	objectNewFields			\
	NP_ITEM_NEW_ONLY	item;

typedef struct NP_ITEM_NEW {

	npItemNewFields

} NP_ITEM_NEW, *P_NP_ITEM_NEW;


/****************************************************************************
 msgNPItemGetPenStyle				takes P_U32, returns STATUS
	Get the pen style of an item.  (Pen styles are defined in notepapr.h.)
*/
#define msgNPItemGetPenStyle	MakeMsg(clsNPItem, 35)

/****************************************************************************
 msgNPItemDelete				takes pNull, returns STATUS
	Delete item from its data.

 Deleting an item decrements its reference count and can cause the
 item to be destroyed.  To prevent, call msgNPItemHold before calling
 msgNPItemDelete.  Then call msgNPItemRelease after working with the
 item.
*/
#define msgNPItemDelete	MakeMsg(clsNPItem, 11)

/****************************************************************************
 msgNPItemPaintBackground	takes P_NP_ITEM_DC, returns STATUS
	Paints a gray background if the receiver is selected.

 Subclasses should override this message if they want a different
 type of selection feedback.
*/
#define msgNPItemPaintBackground	MakeMsg(clsNPItem, 41)

typedef struct {

	OBJECT		dc;			// DC to paint into
	OBJECT		dcPen;		// equivalent DC in pen units

} NP_ITEM_DC, *P_NP_ITEM_DC;


/****************************************************************************
 msgNPItemSelect				takes BOOLEAN, returns STATUS
	Selects or deselects item.
*/
#define msgNPItemSelect	MakeMsg(clsNPItem, 14)

/****************************************************************************
 msgNPItemSelected			takes P_BOOLEAN, returns STATUS
	Passes back item's selection status.
*/
#define msgNPItemSelected	MakeMsg(clsNPItem, 15)

/****************************************************************************
 msgNPItemMove			takes P_XY32, returns STATUS
	Moves item to the indicated position.
*/
#define msgNPItemMove	MakeMsg(clsNPItem, 5)

/****************************************************************************
 msgNPItemDelta			takes P_XY32, returns STATUS
	Moves item by the indicated amount.
*/
#define msgNPItemDelta	MakeMsg(clsNPItem, 6)

/****************************************************************************
 msgNPItemGetViewRect			takes P_RECT32, returns STATUS
	Passes back receiver's bounding rectangle.
*/
#define msgNPItemGetViewRect	MakeMsg(clsNPItem, 19)

/****************************************************************************
 msgNPItemHitRect			takes P_RECT32, returns STATUS
	Returns stsOK if receiver's bounds overlaps pArgs.
*/
#define msgNPItemHitRect	MakeMsg(clsNPItem, 9)

/****************************************************************************
 msgNPItemGetMetrics				takes P_NP_ITEM_METRICS, returns STATUS
	Gets the item's metrics.
*/
#define msgNPItemGetMetrics	MakeMsg(clsNPItem, 20)

typedef struct NP_ITEM_METRICS {

	U8			selected:	1,	// is item selected?
				marked:		1,	// is item marked (in the clsMark sense)?
				reserved:	6;
	U8			refCount;		// number external references to item
								// (not generally interesting to subclasses)
	XY16		baseline;		// item's horizontal and vertical baseline
								// (currently only the y value is used)
	RECT32		bounds;			// window relative bounds
								//     (with respect to its bounds' origin)
	OBJECT		data;			// data object that item is in
	OBJECT		adjunct;		// see msgNPItemSetAdjunct for more information
	U32			penStyle;		// item's pen style

} NP_ITEM_METRICS, *P_NP_ITEM_METRICS;

/****************************************************************************
 msgNPItemSetBaseline				takes P_XY32, returns STATUS
	Sets receiver's baseline.
*/
#define msgNPItemSetBaseline		MakeMsg(clsNPItem, 21)

/****************************************************************************
 msgNPItemSetBounds				takes P_RECT32, returns STATUS
	Sets receiver's bounds.
*/
#define msgNPItemSetBounds		MakeMsg(clsNPItem, 30)


/****************************************************************************
 msgNPItemHold			takes NULL, returns STATUS
	Increments the reference count for the item.

 When the reference count for an item drops to zero, it is destroyed.
*/
#define msgNPItemHold	MakeMsg(clsNPItem, 22)

/****************************************************************************
 msgNPItemRelease			takes NULL, returns STATUS
	Decrements the reference count for the item.

 When the reference count for an item drops to zero, it is destroyed.
*/
#define msgNPItemRelease	MakeMsg(clsNPItem, 23)

/****************************************************************************
 msgNPItemAlignToBaseline		takes P_XY32, returns STATUS
	Moves item so that it align to passed in line spacing.

 The item should be aligned against the y-value of pArgs.
*/
#define msgNPItemAlignToBaseline	MakeMsg(clsNPItem, 33)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Messages that are usually overridden by subclasses.                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNPItemPaint				takes P_NP_ITEM_DC, returns STATUS
	Paints item using the passed in drawing contexts.
*/
#define msgNPItemPaint	MakeMsg(clsNPItem, 12)

/****************************************************************************
 msgNPItemSetPenStyle				takes U32, returns STATUS
	Sets the item's pen style.  (Pen styles are defined in notepapr.h.)
*/
#define msgNPItemSetPenStyle	MakeMsg(clsNPItem, 34)


/****************************************************************************
 msgNPItemSetOrigin			takes P_XY32, returns STATUS
	Set receiver's origin.
*/
#define msgNPItemSetOrigin	MakeMsg(clsNPItem, 18)

/****************************************************************************
 msgNPItemScratchOut			takes P_RECT32, returns STATUS
	Handles the scratch-out gesture on an item.

 Scribble items handle this message by deleting strokes that overlap pArgs.
 Other items simply delete themselves.
*/
#define msgNPItemScratchOut	MakeMsg(clsNPItem, 24)

/****************************************************************************
 msgNPItemSplitGesture		takes P_XY32, returns STATUS
	Handles the split gesture on an item.

 The pArgs refers to the "hot point" for the gesture.
*/
#define msgNPItemSplitGesture	MakeMsg(clsNPItem, 25)

/****************************************************************************
 msgNPItemSplit				takes NULL, returns STATUS
	Split an item into its constituent items.
*/						   
#define msgNPItemSplit		MakeMsg(clsNPItem, 26)

/****************************************************************************
 msgNPItemSplitAsWords		takes NULL, returns STATUS
	Splits receiver into words.  Deletes receiver, inserts new items.

 Return Value
	stsItemNoSplit:		Returned if nothing was split.
*/
#define msgNPItemSplitAsWords		MakeMsg(clsNPItem, 16)


/****************************************************************************
 msgNPItemJoin				takes OBJECT, returns STATUS
	Joins receiver and OBJECT and deletes OBJECT.
*/
#define msgNPItemJoin			MakeMsg(clsNPItem, 27)

/****************************************************************************
 msgNPItemTie					takes OBJECT, returns STATUS
	Joins OBJECT and receiver and deletes them.  Inserts new object.
*/
#define msgNPItemTie			MakeMsg(clsNPItem, 17)

/****************************************************************************
 msgNPItemGetScribble			takes P_OBJECT, returns STATUS
	Pass back the item's scribble.

 Subclasses that do not contain a scribble should not respond to this
 message.
*/
#define msgNPItemGetScribble		MakeMsg(clsNPItem, 4)

/****************************************************************************
 msgNPItemGetString				takes PP_STRING, returns STATUS
	Passes back the text string for the item.

 Subclasses that do not have a text representation should not respond to
 this message.

 clsNPScribbleItem responds to this message by translating its scribble
 and returning the resulting string.

 The sender of this message should either use the passed back string
 immediately or make a copy of it.
*/
#define msgNPItemGetString		MakeMsg(clsNPItem, 38)

/****************************************************************************
 msgNPItemSetString				takes P_STRING, returns STATUS
	Sets the text string for the item.

 Not all items can handle this message.
*/
#define msgNPItemSetString		MakeMsg(clsNPItem, 42)

/****************************************************************************
 msgNPItemToText				takes P_OBJECT, returns STATUS
	Item converts itself to a text item, passes back text item.

 Receiver deletes itself from its data and inserts the text item.
 If pArgs is pNull, the text item is not passed back.
*/
#define msgNPItemToText		MakeMsg(clsNPItem, 7)

/****************************************************************************
 msgNPItemToScribble			takes P_ARGS, returns STATUS
	Item converts itself to a scribble item.

 Receiver deletes itself from its data and inserts the scribble item.
*/
#define msgNPItemToScribble	MakeMsg(clsNPItem, 36)

/****************************************************************************
 msgNPItemHitRegion			takes P_RECT32, returns STATUS
	Returns stsOK if receiver's path overlaps pArgs.
*/
#define msgNPItemHitRegion	MakeMsg(clsNPItem, 10)

/****************************************************************************
 msgNPItemCalcBaseline		takes P_XY32, returns STATUS
	Calculate and set receiver's baseline.

 The calculation is based on the line spacing specified by pArgs.
*/
#define msgNPItemCalcBaseline	MakeMsg(clsNPItem, 28)

/****************************************************************************
 msgNPItemCalcBounds			takes OBJECT, returns STATUS
	Receiver calculates and sets its new bounds.

 Usually send in response to the item's style changing.  OBJECT is the
 data object in which the item will be inserted.  If the item is in a
 data object, pArgs can be pNull.
*/														 
#define msgNPItemCalcBounds	MakeMsg(clsNPItem, 37)

/****************************************************************************
 msgNPItemGetWordSpacing		takes P_U16, returns STATUS
	Receiver passes back the size of its "space" character.

 This message is used by msgNotePaperTidy to determine the spacing
 of items.
*/
#define msgNPItemGetWordSpacing	MakeMsg(clsNPItem, 43)

/****************************************************************************
 msgNPItemCanBeTranslated		takes pNull, returns STATUS
	Receiver returns stsOK if it can be translated.

 Translation occurs in response to msgNPItemToText.
*/
#define msgNPItemCanBeTranslated	MakeMsg(clsNPItem, 13)

/****************************************************************************
 msgNPItemCanBeUntranslated		takes pNull, returns STATUS
	Receiver returns stsOK if it can be untranslated.

 Untranslation occurs in response to msgNPItemToScribble.
*/
#define msgNPItemCanBeUntranslated	MakeMsg(clsNPItem, 31)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Private messages.                                                      *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNPItemHasString				takes pNull, returns STATUS
	Returns stsOK if the item has a string.

 Text items always respond to this message with stsOK.

 Scribble items will only respond with stsOK if they have been
 translated but are currently being displayed as a scribble.
 In contrast, msgNPItemGetString causes a scribble item
 to be translated.
*/
#define msgNPItemHasString		MakeMsg(clsNPItem, 1)

/****************************************************************************
 msgNPItemCanJoin			takes OBJECT, returns STATUS
	Subclasses should return stsOK of they can join with OBJECT.

 Returning stsOK to this message implies a positive response to
 msgNPItemJoin given the same argument.

 Scribble items can join with other items that can return a scribble
 via msgNPItemGetScribble.  Text items can join with other items that
 have a string.
*/
#define msgNPItemCanJoin		MakeMsg(clsNPItem, 2)

/****************************************************************************
 msgNPItemSetDataObject				takes OBJECT, returns STATUS
	Sets receiver's data (e.g. instance of clsNPData).
*/
#define msgNPItemSetDataObject			MakeMsg(clsNPItem, 29)

/****************************************************************************
 msgNPItemSetAdjunct				takes OBJECT, returns STATUS
	Sets receiver's adjunct item.

 When an instance of clsNPScribbleItem is translated, the original
 scribble is stored in the resulting text item's adjunct field.  This is
 done to support "untranslation".  When the text's pen style or location
 is changed, its adjunct is modified as well.  The work of keeping the
 two items synchronized is handled by clsNPItem.
*/
#define msgNPItemSetAdjunct		MakeMsg(clsNPItem, 32)


/****************************************************************************
 msgNPItemMark				takes P_U32, returns STATUS
	Receiver marks itself and provides its mark token.

 If the item is unmarked, it gets a unique mark id from the data.
 Otherwise, it increments its mark reference count.

 This message is used by clsNotePaper in conjunction with the clsMark protocol.
*/
#define msgNPItemMark			MakeMsg(clsNPItem, 8)

/****************************************************************************
 msgNPItemUnmark				takes P_ARGS, returns STATUS
	Receiver unmarks itself and passes back the mark id if applicable.

 If the item's mark reference count is 1, it unmarks itself and
 returns its token to its data object for reuse.
 Otherwise, it just decrements its mark reference count.

 This message is used by clsNotePaper in conjunction with the clsMark protocol.
*/
#define msgNPItemUnmark		MakeMsg(clsNPItem, 40)

/****************************************************************************
 msgNPItemGetMarkId				takes P_U32, returns STATUS
	Passes back the mark id of an item.

 This message is used by clsNotePaper in conjunction with the clsMark protocol.
*/
#define msgNPItemGetMarkId	MakeMsg(clsNPItem, 39)

#endif
