/****************************************************************************
 gwin.h

 Copyright 1992, GO Corporation, All Rights Reserved

 $Revision:   1.10  $
   $Author:   kbentley  $
	 $Date:   09 Mar 1992 16:36:10  $

 This file contains the API definition for clsGWin.

 clsGWin inherits from clsWin.
****************************************************************************/


/**** Introduction ****/
/*
 clsGWin provides a convenient default implementation of several important
 PenPoint features -- gesture and keyboard processing, quick help
 interaction and event forwarding.

 clsGWin is an ancestor of many of PenPoint's window-based classes,
 including all of the Toolkit classes.

 Many tasks involving the input system and the handwriting recognition
 system can be handled very simply using only a few clsGWin messages.  Some
 tasks require use of some of clsGWin's more sophisticated messages.  And
 there are some task for which clsGWin is not appropriate.  For instance,
 even a modest drawing application or "ink editor" will almost certainly
 have to interact more directly with the input system and handwriting
 recognition system.

 Several important task can be accomplished by using just few clsGWin
 messages:

    -:	To process gestures, see msgGWinGesture.

    -:	To process keyboard input, see msgGWinKey.

    -:	To implement quick help, use gWin's helpId;  see GWIN_NEW_ONLY and
        msgGWinSetHelpId.

    -:	To process gestures and keyboard events which occurred in child
		windows, see msgGWinForwardedGesture or msgGWinForwardedKey.

    -:	To control whether or not a window responds to gestures, see the
        gestureEnable field in GWIN_STYLE.

 More complex subclasses will need to understand more details, as described
 below.
*/

/**** Debugging Flags ****/
/*
 GWin's debugging flag set is '#' (0x23).  Defined flags are:

	0001:  Display generally useful messages.
	0004:  Display messages during quick help processing.
	0010:  Display messages during timeout processing.
*/

/**** Keyboard Processing ****/
/*
 Keyboard processing and forwarding occurs when a gWin receives
 msgInputEvent with a key event message in pArgs->devCode.  The steps taken
 are:

	-:	gWin self sends msgGWinKey with the event.

    -:	If the response to msgGWinKey is stsRequestDenied, gWin self sends
        msgGWinBadKey.  gWin's default response to msgGWinKey is to return
		stsRequestForward, which causes gWin to perform key event
		forwarding.

    -:	If the response to msgGWinKey is stsRequestForward and
        style.keyboardForward is set, gWin self sends msgGWinForwardKey. 
		In response to this message, the gWin packages up the data and
		uses msgWinSend to forward the key information.  This results in
		parent windows potentially receiving msgGWinForwardedKey
		(see msgGWinForwardedKey description). gWin's default response to
		msgGWinForwardedKey is to return stsRequestForward, which causes
		the event forwarding to continue.

    -:	If the response to msgGWinForwardKey is stsRequestDenied or 
        stsRequestForward, gWin self sends msgGWinBadKey.

*/

/**** Gesture Processing ****/
/*
 A gWin self sends msgGWinGesture when one of the following occurs.  (Each
 of these is described more detail below.)

    -:	Case 1:  A gWin receives msgGWinXList (typically because a
        translation has completed).

    -:	Case 2:  A gWin receives msgInputEvent with an event of press-hold or
        a tap-press-hold.

    -:	Case 3:  A gWin receives msgQuickHelpHelpShow from
        theQuickHelpManager.
 
 If the response to msgGWinGesture is stsRequestDenied, the gesture is
 unrecognized and one of the following actions is taken:
	
    -:	In Case 1, a translated gesture, msgGWinBadGesture is self sent.
 
    -:	In Case 2, normal gesture processing continues.  This is because
        a press-hold or a tap-press-hold gesture is sent in response to an
        input event while potentially in the process of collecting data for
        another gesture (see below).
 
    -:	In Case 3, the "no help available" help is displayed via
        msgQuickHelpShow. 
 
 If the response to msgGWinGesture is stsRequestForward,
 msgGWinForwardGesture is self sent.  If the response to
 msgGWinForwardGesture is stsRequestDenied or stsRequestForward, the same
 action is taken as if msgGWinGesture returned stsRequestDenied.


 Case 1:  How a GWin Receives Translated Gestures.

 msgGWinGesture is self sent in response to msgGWinXList.
 msgGWinXList is self sent by gWin after an xGesture translator has
 completed its translation.  This occurs as follows:

 When msgPenStroke is received from the input system, the gWin adds strokes
 to a gesture translator.  This is done via a self send of msgGWinStroke,
 which adds the stroke via sending msgScrAddStroke to the gesture
 translator.  

 	gWin 	--> msgScrAddStroke		--> xGesture Translator

 When an "out of proximity" event is received, gWin self sends
 msgGWinComplete.  In response to the msgGWinComplete, gWin sends
 msgScrComplete to the gesture translator. 
 
 	gWin  	--> msgScrComplete		--> xGesture Translator 

 The translator then sends msgXlateCompleted back to the gWin, indicating
 translation is complete.  GWin retrieves translated results by sending
 msgXlateData to the gesture translator.

 	gWin	<-- msgXlateCompleted	<-- xGesture Translator
	gWin	--> msgXlateData		--> xGesture Translator

 This returns an xlist containing the translated data (see xlist.h).  GWin
 then self sends msgGWinXList to process the xlist.  This extracts the
 appropriate information from the xlist (via XList2Gesture).  gWin then
 performs the gesture processing and forwarding described below: 

    -:	Self send msgGWinGesture.

    -:	If msgGWinGesture returns stsRequestDenied, gWin self sends
        msgGWinBadGesture.

    -:	If msgGWinGesture returns stsRequestForward and style.gestureForward
        is set, gWin self sends msgGWinForwardGesture.  Similar to the
		forwarding of keyboard events, the gWin packages up the gesture
		information and uses msgWinSend to forward the gesture.  This
		results in parent windows potentially receiving
		msgGWinForwardedGesture (see msgGWinForwardedGesture).

    -:  If msgGWinForwardGesture returns stsRequestForward and the gesture
        is the help gesture, gWin calls PenPoint's quick help with
        hlpQuickHelpNoHelp. This invokes quick help with the "No help
        available" text.

	-:	If msgGWinForwardGesture returns stsRequestDenied or 
		stsRequestForward, gWin self sends msgGWinBadGesture


 Case 2:  How a GWin Synthesizes Some Gestures.

 If, when processing input events, gWin sees a press-hold or a
 tap-press-hold input event, gesture processing and forwarding takes place.
 If the gesture is unrecognized, then normal input processing continues.
 This means that if an end-user press-holds on an area where press-hold has
 no meaning, the window in question receives msgGWinGesture with
 xgsPressHold. The window returns stsRequestForward (as will all the
 windows that see msgGWinForwardGesture).  Normal processing continues, and
 when the user lifts the pen the translation of the single tap occurs and
 the gesture processing mentioned above takes place.  If the gesture is
 recognized, the gesture translation is aborted and input data is thrown
 away until (and including) the next Pen Up event.  A description:
 	
    -:	gWin self sends msgGWinGesture with xgsPressHold or xgsTapHold.

    -:	If the response to msgGWinGesture is stsRequestDenied, processing of
        the input continues.

    -:	If the response to msgGWinGesture is stsOK, gesture processing is
        aborted.

    -:	If the response to msgGWinGesture is stsRequestForward and
        style.gestureForward is set, gWin self sends msgGWinForwardGesture.
        This results in parent windows potentially receiving 
        msgGWinForwardedGesture (see msgGWinForwardedGesture).

    -:	If the response to msgGWinForwardGesture is stsRequestDenied or 
        stsRequestForward, processing of the input continues.

    -: 	If the response to msgGWinForwardGesture is stsOK, gesture
        processing is aborted.


 Case 3:  How a GWin Responds to msgQuickHelpHelpShow.

 The final case in which msgGWinGesture is sent is in response to 
 msgQuickHelpHelpShow. This is sent from theQuickHelpManager when in help
 mode and the user taps on the screen. GWin responds by sending
 msgGWinGesture with the help gesture, and performing similar forwarding
 above. When msgGWinGesture returns stsRequestDenied, or 
 msgGWinForwardGesture returns stsRequestDenied or stsRequestForward, gWin
 sends msgQuickHelpShow to display the No Help Available message.
*/

/**** style.gestureLocal and Coordinate Transformations ****/
/*
 When using large windows (width or height near or above 2^16), you should
 style.gestureLocal to true.  Doing so avoids some potential numeric
 overflow conditions that can make gesture recognition unreliable.

 Setting style.gestureLocal true changes the coordinate system used 
 internally by gWin.  It also changes the coordinate system used in some of
 gWin's more sophisticated self sent messages.  If you don't use these more
 sophisticated messages, you can just set style.gestureLocal true and never
 worry about it again, regardless of the size of your window.  If you do
 use these messages, then you should read the rest of this section to
 understand what's different.

 Here are the messages whose parameters are affected by style.gestureLocal:
	-:	msgGWinStroke
	-:	msgGWinXList
	-:	msgGWinTransformGesture
	-:	msgGWinTransformXList

 Normal gesture processing (style.gestureLocal is false) is done using the
 following coordinate transformations:

    -:	The stroke input event is delivered with pArgs->xy set to the local
        window coordinates and the pen data in root window pen coordinates.

    -:	On the first stroke to the window, gWin remembers an offset of
        (0,0).  This step is obviously trivial in this case but is
        important when style.gestureLocal is true.

    -:	This value is first converted to root window coordinates and then the
        resulting value is converted to pen units.

    -:	This vector is subtracted from the origin of the pen stroke data.
        The pen stroke data is still in pen units but has been shifted so
        that its origin is relative to the local window origin.

    -:	This shifted stroke is self sent using msgGWinStroke.  THIS IS
        IMPORTANT.  Any object intercepting this message gets pen data that
        has been shifted to appear in the local window.  This is slightly
        different than the pen stroke which comes from the input system.

    -:	In response to the first msgGWinStroke, gWin creates a translator
        and makes itself an observer of the translator.  The stroke is then
        added to the translator.

    -:	Normal input collection of strokes continues.  Eventually the
        gesture is completed and translation occurs.
	
    -:	In response to msgXlateComplete, gWin gets the XList data and
        converts it from pen units to window units.  Remember that since
        the pen strokes were shifted by the origin of the window (in
        digitizer units), the window units give locations in the local
        window.  gWin then self sends msgGWinXList.

    -:	In response to msgGWinXList, gWin converts the xlist information and
        self sends msgGWinGesture.

 If style.gestureLocal is true, the same sequence of events occurs, but
 with the following change in coordinate systems:

    -:	When the first stroke comes in to the gWin, the local window
        coordinates of the stroke are saved as the offset instead of 0,0. 
        This value is converted to root window coordinates and then
        converted to pen units and used to offset the stroke.  This means
        that the stroke is no longer in local window space, but rather are
        in root window coordinate space.
		
    -:	Subclasses which handle msgGWinStroke are getting data which is root
        window relative.  If they need it in local window space then they
        have to transform it first.

    -:	When the translation is complete, the offset that was remembered
        earlier is converted to root window coordinates and then to pen
        units.  This offset is added to the points returned by the
        translator before converting back to screen units.  The effect is
        that now the gesture is shifted back to its proper location in root
        window space.

    -:	When converting the XList data to the GWin gesture, the important
        points are converted from root window coordinates back to local
        window coordinates before self sending msgGWinGesture.
*/
#ifndef GWIN_INCLUDED
#define GWIN_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

#ifndef CLSMGR_INCLUDED
#include <clsmgr.h>
#endif

#ifndef WIN_INCLUDED
#include <win.h>
#endif

// Next up:	25	Recycled: 3

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT GWIN;

/****  Default Window Flags  ****/
/*
 * These are the default input flags set by a gWin at msgNew time if
 * gestureEnable is set. Changing these flags after new time is possible, but
 * extreme care needs to be taken as these define the pen events that get
 * generated to the window.
*/
#define gWinInputFlags	(inputStroke | inputOutProx | \
						inputInk | inputTip |inputTimeout | inputAutoTerm | \
						inputEnter | inputHoldTimeout)

/****  Style Structure  ****/
typedef struct GWIN_STYLE {

	U16 gestureEnable:		1,		// enables gesture translation 
	    gestureForward:		1,		// enables forwarding of gestures
	    gestureLocal:		1,		// enables localized strokes for large
									// gesture windows (>32K digitizer pts)
        keyboardForward:	1,		// enables forwarding of key events
	    privateData1:		2,		// private
		grabDown:			1,		// grab input on msgPenDown vs. 
									// msgPenStroke
		grabActive:			1,		// private
		firstEnter: 		1,		// grab on msgPenEnter if no other grab
		tossingEvents:		1,		// private
		askOtherWin:		1,		// ask other gWin if it wants event
		otherWinSaysYes:	1,		// answer yes if asked if you want event
	    reserved:   		4;		// reserved for future use

} GWIN_STYLE, *P_GWIN_STYLE;

/****  Gesture Structure  ****/
/*
 * This data structure defines all information returned by a gesture
 * translator in the form of a simple data structure.  It is used as a
 * parameter to many of the gesture methods defined in gWin.
*/
typedef struct GWIN_GESTURE {
	MESSAGE	 	msg;		// gesture Id
	RECT32	  	bounds;	 	// bounding box in LWC
	XY32		hotPoint;   // gesture hot point
	OBJECT		uid;		// object in which the gesture was generated
	U32			reserved;	// reserved for future use
} GWIN_GESTURE, *P_GWIN_GESTURE;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages											   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_GWIN_NEW, returns STATUS
	category: class message
	Creates and initializes a new instance.  

 If gWin.style.gestureEnable is true, then gWin ORs in gWinInputFlags into
 pArgs->win.flags.input before passing the message to its ancestors.  These
 win.flags.input bits can be changed after the gWin is created, but extreme
 care should be taken!

 If setting a helpId, setting the pNew->gWin.helpId to the same as the 
 pNew->win.tag helps minimize memory needed by the object.  It is
 recommended that the helpId be the same as the window tag if possible.
 However, if the window tag changes when the help id is the same as the
 window tag, then the help tag will change too.
*/
typedef struct GWIN_NEW_ONLY {
	GWIN_STYLE style;			// gWin style flags
	U32 helpId;					// quick help id
	U32 reserved;
} GWIN_NEW_ONLY, *P_GWIN_NEW_ONLY;

#define gWinNewFields			\
	winNewFields				\
	GWIN_NEW_ONLY	   gWin;

typedef struct GWIN_NEW {
	gWinNewFields
} GWIN_NEW, *P_GWIN_NEW;

/****************************************************************************
 msgNewDefaults		takes P_GWIN_NEW, returns STATUS
    category: class message
	Initializes the GWIN_NEW structure to default values.

 Zeros out pNew->gWin and sets:
//{
	pArgs->gWin.style.gestureEnable = TRUE;
	pArgs->gWin.style.gestureForward = TRUE;
	pArgs->gWin.style.keyboardForward = TRUE;
	pArgs->gWin.style.grabDown = TRUE;
	win.input = gWinInputFlags;
//}
*/

/****************************************************************************
 msgGWinGetStyle	takes P_GWIN_STYLE, returns STATUS
	Returns the current style.
*/
#define msgGWinGetStyle				MakeMsg(clsGWin, 12)

/****************************************************************************
 msgGWinSetStyle	takes P_GWIN_STYLE, returns STATUS
	Sets the style settings.

 If gestureEnable is true, gWin ORs in the gWinInputFlags with the window
 flags.  (See the comments near msgNew in this file.)  Setting gestureEnable
 to false does NOT clear these flags.
*/
#define msgGWinSetStyle				MakeMsg(clsGWin, 13)

/****************************************************************************
 msgGWinSetHelpId	takes U32, returns STATUS
	Sets the gWin's helpId for quick help.

 Setting the helpId to be identical to the gWin's win.tag helps minimize
 the amount of instance data taken by a gWin.
*/
#define msgGWinSetHelpId			MakeMsg(clsGWin, 16)

/****************************************************************************
 msgGWinGetHelpId	takes P_U32, returns STATUS
	Returns the gWin's helpId.
*/
#define msgGWinGetHelpId			MakeMsg(clsGWin, 17)

/****************************************************************************
 msgGWinGetTranslator	takes P_OBJECT, returns STATUS
	Returns the gWin's translator object.

 gWin's default response is to return the current translator object.
 
 By default, gWin has a null current translator unless strokes have been
 added since msgNew or since the last msgGWinAbort or msgGWinComplete.  (In
 other words, gWin does not have a translator unless it is currently
 collecting or translating strokes.)

 See Also
	msgGWinAbort
	msgGWinComplete
*/
#define msgGWinGetTranslator		MakeMsg(clsGWin, 7)

/****************************************************************************
 msgGWinSetTranslator	takes P_OBJECT, returns STATUS
	Sets the translator object and returns the previous one.

 This message has no affect if the gWin has not received a stroke from the
 input system since the last msgGWinComplete or msgGWinAbort.
 
 Because of this limitation you probably should not use this message.

 gWin's default response is to set its translator object to pArgs AND to
 set *pArgs to the uid of the previous translator.

 See Also
	msgGWinAbort
	msgGWinComplete
*/
#define msgGWinSetTranslator		MakeMsg(clsGWin, 8)

/****************************************************************************
 msgGWinTransformGesture	takes P_GWIN_GESTURE, returns STATUS
	Transforms gesture information into local window coordinates.

 This message is useful for clients who handle msgGWinForwardedGesture.

 Transforms the gesture bounds and hotPoint into the local window coordinate
 system.  
 
 This is only necessary if the gesture occurred in a window other
 then self. 
 
 gWin's default response modifies the bounds, hotPoint, and uid (set to
 self) fields. 
*/
#define msgGWinTransformGesture		MakeMsg(clsGWin, 14)

/****************************************************************************
 msgGWinTransformXList	takes P_XLIST, returns STATUS
	Transforms xlist information to local window coordinates.

 This message is useful for clients who handle msgGWinXList.
 
 This message is only necessary if the xlist was generated relative to a
 window other then self.  This message transforms the gesture bounds and
 hotPoint to local window coordinates system.
*/
#define msgGWinTransformXList		MakeMsg(clsGWin, 15)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Gesture Processing							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgGWinStroke		takes P_INPUT_EVENT, returns STATUS
	Self sent to process a pen stroke received from the input system.

 If style.gestureEnable is false, gWin's default response is to return
 stsOK.

 If style.gestureEnable is true, gWin's default response is as follows.
 First, if the gWin has no translator, one is created by self sending 
 msgGWinTranslator and gWin makes itself an observer of the translator.
 Next It then sends msgScrAddedStroke to the translator to tell the
 translator that the gWin has received a new stroke.

 Subclasses can handle this message and process individual strokes.  If
 style.gestureLocal is false, stroke coordinates are self relative;  if
 style.gestureLocal is true, stroke coordinates are root window relative.

 See Also
	msgGWinTranslator
*/
#define msgGWinStroke				MakeMsg(clsGWin, 5)

/****************************************************************************
 msgGWinTranslator		takes P_OBJECT, returns STATUS
	Self sent to retrieve the translator used to gather and translate strokes.

 gWin's default response is to create an instance of clsXGesture.

 gWin self sends msgGWinTranslator whenever it needs a translator to gather
 and translate strokes.  For instance, when gWin receives msgGWinStroke,
 and the stroke is the first stroke in a new gesture, gWin self sends 
 msgGWinTranslator.

 The translator will be destroyed during gWin's handling of msgGWinComplete
 or msgGWinAbort.

 See Also
	msgGWinComplete
	msgGWinAbort
*/
#define msgGWinTranslator			MakeMsg(clsGWin, 4)

/****************************************************************************
 msgGWinComplete	takes void, returns STATUS
	Self sent to complete a gesture.

 gWin self sends msgGWinComplete when "out of proximity" or a timeout
 occurs.  Clients can send msgGWinComplete to cause gesture completion
 and translation.

 gWin's default response to is cause translation as described in the
 introductory material at the beginning of this file.  gWin then destroys 
 its translator.

 If the gWin has a grab (perhaps because it was collecting strokes when a
 client sends this message), the grab is NOT terminated in response to this
 message.  But the gWin will remember that this message has been received
 and will terminate the grab in response to the next msgInputEvent it
 receives.
*/
#define msgGWinComplete				MakeMsg(clsGWin, 6)

/****************************************************************************
 msgGWinAbort	takes void, returns STATUS
	Aborts a gesture.

 gWin's default response is very similar to its response to
 msgGWinComplete, except that the translation is aborted instead
 completed.  As with msgGWinComplete, the gWin destroys its translator and
 ceases collecting strokes.
 
 A client can send msgGWinAbort to abort the gWin's processing of a
 gesture.

 The grab behavior is identical to that described with msgGWinComplete.

 Subclasses may field msgGWinAbort but must also allow their ancestor to
 see the message.
*/
#define msgGWinAbort				MakeMsg(clsGWin, 9)

/****************************************************************************
 msgGWinXList	takes P_XLIST, returns STATUS
	Self sent by gWin to process an xlist.

 After a translation has been completed (in other words, after gWin has
 received msgXlateCompleted from its translator), gWin extracts the
 translation data (in the form of an xlist) from the translator, and then
 self sends msgGWinXList.

 gWin's default response is to extract the gesture information from the
 xlist (using the xlist utility routine XList2Gesture) and then self
 sends msgGWinGesture.

 See Also
	msgGWinGesture
	msgGWinTransformXList
*/
#define msgGWinXList				MakeMsg(clsGWin, 1)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Gesture Recognition and Forwarding Messages			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgGWinGesture		takes P_GWIN_GESTURE, returns STATUS
	Self-sent to process a gesture.

 The default response to msgGWinGesture is as follows:

 For the help gesture(s), return the result of self sending msgGWinHelp. 
 By default, msgGWinHelp returns stsRequestForward if the helpId is zero,
 or stsOK if there is a valid helpId.  
 
 For all other gestures, return stsRequestForward.
 
 Effectively, the default response of gWin to msgGWinGesture is to return
 stsOK if the gesture is a help gesture on a window and the window has a
 valid helpId.  Otherwise the default behavior is to return
 stsRequestForward.

 GWin's default response to msgGWinForwardedGesture is the same as
 msgGWinGesture.  This means that the help gesture(s) is forwarded up the
 window hierarchy until a gWin has a valid helpId, and then that gWin 
 sends the appropriate message and quick help id to theQuickHelpManager.
 
 Hence a window can have a common helpId (and corresponding help text) for
 all (or some) child windows, and the quick help text displayed will be the
 same regardless of the child window the gesture actually occurred in.
 
 Return Value
	stsRequestForward:	The gesture was not processed and should be 
						forwarded.
	stsRequestDenied:	The gesture was not processed and should not be
						forwarded.  
	stsOK:				The gesture was processed and should not be
						forwarded.

 See Also
	msgGWinXList
	msgGWinHelp
*/
#define msgGWinGesture				MakeMsg(clsGWin, 2)

/****************************************************************************
 msgGWinForwardGesture		takes P_GWIN_GESTURE, returns STATUS
	Causes a gesture to be forwarded to parent windows.  

 Subclasses should not handle this message.

 In response to this message, gWin initiates gesture forwarding.  This
 results in each parent window within the same process receiving
 msgGWinForwardedGesture, from the immediate parent to the root.  
 
 If any window along the path returns stsOK from msgGWinForwardedGesture,
 or the window has style.gestureForward off, stsOK is returned.  
 
 gWin performs this forwarding via msgWinSend.  The status returned to the
 sender of msgGWinForwardGesture is the status returned by this msgWinSend.
 See the comments for msgGWinForwardedGesture for return values and their
 interpretation.

 The msgWinSend of msgGWinForwardedGesture is only delivered to windows in
 same process.

 Return Value
	stsRequestForward:	The gesture was not processed by any of the 
						ancestor windows.  Further processing should
						occur if possible.
	stsRequestDenied:	The gesture was not processed by any of the
						ancestor windows, and was aborted at some
						level of the walk. No further processing should
						occur.
	stsOK:				The gesture was processed.  No further processing
						should occur.

 See Also
	msgGWinXList
	msgWinSend
*/
#define msgGWinForwardGesture		MakeMsg(clsGWin, 20)

/****************************************************************************
 msgGWinForwardedGesture	takes P_GWIN_GESTURE, returns STATUS
	Message received when a gesture is forwarded.

 See the comments describing msgGWinGesture.

 msgGWinForwardedGesture is sent to a gWin when a gesture event has been
 forwarded from a child window.  Subclasses wishing to process gestures
 forwarded from child windows should handle this message.

 Do not send this message;  it should only be self sent by clsGWin.

 Return Value
	stsRequestForward:	The gesture was not processed and should be 
						forwarded further.
	stsRequestDenied:	The gesture was not processed and should not be
						forwarded any further.  
	stsOK:				The gesture was processed and should not be
						forwarded any further.

 See Also
	msgGWinHelp
	msgGWinForwardGesture
	msgGWinTransformGesture
*/
#define msgGWinForwardedGesture		MakeMsg(clsGWin, 11)

/****************************************************************************
 msgGWinBadGesture		takes P_GWIN_GESTURE, returns STATUS
	Displays feedback for unrecognized and unknown gestures.  

 gWin's response is to display the unrecognized gesture feedback (if 
 pArgs->msg == xgsNull) or the unknown gesture feedback (for any other
 value of pArgs->msg).

 gWin's default response to msgGWinXList includes self-sending
 msgGWinBadGesture if the gesture is unrecognized by the recognition system
 (xgsNull) or if none of the recipients of msgGWinGesture and
 msgGWinForwardedGesture processed the gesture.

 See Also
	msgGWinXList
*/
#define msgGWinBadGesture			MakeMsg(clsGWin, 10)

/****************************************************************************
 msgGWinHelp	takes NULL, returns STATUS
	The gWin displays quick help for itself.

 If the gWin's helpId is 0, gWin returns stsRequestForward.  Otherwise,
 gWin sends msgQuickHelpShow to theQuickHelpManager with the gWin's help
 id.

 Return Value
	stsRequestForward:	The helpId of self is 0.
	stsOK:				Quick help was invoked for self.
*/
#define msgGWinHelp                 MakeMsg(clsGWin, 22)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Keyboard Processing and Forwarding Messages			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgGWinKey		takes P_INPUT_EVENT, returns STATUS
	Self sent to process a key input event.  

 As part of its default response to msgInputEvent, gWin self sends
 msgGWinKey if the input event is a key event.

 gWin's default response to msgGWinKey is to return stsRequestForward.

 Return Value
	stsRequestForward:	The key event was not processed and should be 
						forwarded further.
	stsRequestDenied:	The key event was not processed and should not be
						forwarded any further.  
	stsOK:				The key event was processed and should not be
						forwarded.
*/
#define msgGWinKey					MakeMsg(clsGWin, 21)

/****************************************************************************
 msgGWinForwardKey		takes P_INPUT_EVENT, returns STATUS
	Initiates keyboard event forwarding.

 Subclasses should not handle this message.

 In response this message, gWin initiates keyboard event forwarding.
 This results in each parent window within the same process receiving
 msgGWinForwardedKey, from the immediate parent to the root.

 If any window along the path returns stsOK from msgGWinForwardedGesture,
 or the window has style.keyboardForward off, stsOK is returned.  

 gWin performs this forwarding via msgWinSend.  The status returned to the
 sender of msgGWinForwardKey is the status returned by this msgWinSend. See
 the comments for msgGWinForwardedKey for return values and their
 interpretation.

 The msgWinSend of msgGWinForwardedKey is only delivered to windows in same
 process.

 Return Value
	stsRequestForward:	The key event was not processed by any of the 
						ancestor windows, and should be forwarded further
						if meaningful.
	stsRequestDenied:	The key event was not processed by any of the
						ancestor windows, and was aborted at some
						level of the walk. No further processing
						should occur.
	stsOK:				The key event was processed.  No further processing
						should occur.

 See Also
	msgWinSend
*/
#define msgGWinForwardKey			MakeMsg(clsGWin, 19)

/****************************************************************************
 msgGWinForwardedKey	takes P_INPUT_EVENT, returns STATUS
	Message received when a keyboard event is forwarded to a gWin.

 msgGWinForwardedKey is sent to a gWin when a keyboard event has been
 forwarded from a child window.  Subclasses wishing to handle keyboard
 events forwarded from child windows should handle this message.

 gWin's default response is to return stsRequestForward.

 Do not send this message;  it should only be self sent by clsGWin.

 Return Value
	stsRequestForward:	The key event was not processed and should be 
						forwarded further.
	stsRequestDenied:	The key event was not processed and should not be
						forwarded any further.  
	stsOK:				The key event was processed and should not be
						forwarded any further.
*/
#define msgGWinForwardedKey			MakeMsg(clsGWin, 18)

/****************************************************************************
 msgGWinBadKey		takes P_INPUT_EVENT, returns STATUS
	Self sent to allow a subclass to handle bad keys.

 gWin's default response is to return stsOK.

 gWin self sends msgGWinBadKey when (1) msgGWinKey returns
 stsRequestDenied, (2) msgGWinKey returns stsRequestForward and
 style.keyboardForward is not set, or (3) msgGWinForwardKey returns
 stsRequestDenied or stsRequestForward.
*/
#define msgGWinBadKey				MakeMsg(clsGWin, 23)

/****************************************************************************
 msgGWinIsComplete	takes P_GWIN_GESTURE, returns STATUS
	Called to determine if a gesture was sent while processing input.

 This message is used to determine if the gesture may have been sent other
 than when processing msgGWinXList or msgQuickHelpHelpShow. Put simply,
 this message returns stsOK for any gesture other then those sent while
 processing input where gesture processing may continue. Examples are
 press-hold and tap-press hold.

 Return Value
	stsRequestDenied			The gesture was sent while processing input
	stsOK						The gesture was sent from msgGWinXList or
								msgQuickHelpHelpShow.
*/
#define msgGWinIsComplete			MakeMsg(clsGWin, 24)

/****************************************************************************
 msgGWinGestureDone 	takes P_GWIN_GESTURE, returns STATUS
    category: self-sent
    Sent to indicate the end of a gesture.

 As part of its default response to msgGWinXList, gWin self sends
 msgGWinGestureDone.  (msgGWinXList is self sent after the forwarding
 protocol has completed but before msgQuickHelpShow or msgGWinBadGesture is
 sent.)
 
 It is intended for use by classes that modify their state in anticipation
 of receiving msgGWinGesture and fail to receive it.  (For instance, a
 subclass could handle msgGWinGesture and not pass the message along to its
 ancestor).  Such classes should watch for msgGWinAbort and
 msgGWinGestureDone.  Either, but not both, could be sent for any one
 gesture.

 Subclasses may field msgGWinGestureDone but must also allow their ancestor
 to see the message.
*/
#define msgGWinGestureDone			MakeMsg(clsGWin, 25)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages from Other Classes							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFree	takes OBJ_KEY, returns STATUS
	Defined in clsmgr.h.

 In response to msgFree, gWin removes itself as an observer of its 
 translator and then destroys the translator.  In addition, gWin frees any 
 memory it has allocated.
*/

/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Defined in clsmgr.h.

 In response to msgSave, gWin saves state information.  gWin files its
 helpId if the helpId is different then the window tag.

 Note that the gWin's translator is not saved or restored since the 
 translator only exists while the gWin is actively collecting strokes.
*/

/****************************************************************************
 msgRestore		takes P_OBJ_RESTORE, returns STATUS
	Defined in clsmgr.h.

 In response to msgRestore, gWin restores state information, including
 the helpId.

 Note that the gWin's translator is not saved or restored since the 
 translator only exists while the gWin is actively collecting strokes.
*/

/****************************************************************************
 msgWinSend		takes P_WIN_SEND, returns STATUS
	Defined in win.h.

 gWin handles msgWinSend if pArgs->msg is msgGWinForwardedGesture or
 msgGWinForwardedKey.  For all other values of pArgs->msg, gWin simply
 passes the message to its ancestor.

 If pArgs->msg is msgGWinForwardedGesture, gWin self sends
 msgGWinForwardedGesture. If this returns stsRequestForward and the gWin's
 style.gestureForward is set, gWin passes the msgWinSend to its ancestor,
 allowing the forwarding to continue.  Otherwise gWin returns the result of
 the self send of msgGWinForwardedGesture.  
 
 If pArgs->msg is msgGWinForwardedKey, gWin self sends msgGWinForwardedKey.
 The response to this message is handled similarly to the gesture case,
 except that style.keyboardForward is checked rather than
 style.gestureForward.
 
 See Also
	msgGWinForwardKey
	msgGWinForwardGesture
	msgGWinForwardedKey
	msgGWinForwardedGesture
*/

/****************************************************************************
 msgInputEvent	takes P_INPUT_EVENT, returns STATUS
	Defined in input.h.

 This is the main processing message for gWin.  
 
 For keyboard events, gWin self sends msgGwinKey, and performs the keyboard
 processing and forwarding as described earlier. 
 
 For pen events, gWin returns stsInputTerminate if gestureEnable is not
 set. Otherwise, gWin initiates a grab by returning stsInputGrabTerminate
 on msgPenDown if style.grabDown is set.  
 
 On msgPenStroke events gWin self sends msgGWinStroke and continues the
 grab by returning stsInputGrabTerminate.  
 
 On msgPenOutProxUp, msgPenOutProxDown, or msgPenTimeout gWin self sends
 msgGWinComplete and releases the grab by returning stsInputTerminate.
 
 For other pen events, gWin returns stsInputTerminate or stsInputTerminate
 if it "grabbing" (has returned stsInputGrabTerminate due to a msgPenDown
 or msgPenStroke), and not "released-the-grab" (returned stsInputTerminate
 due to a msgPenOutProxDown, msgPenOutProxUp, or msgPenTimeout).
 
 If gWin receives a msgPenTap, is not "grabbing", and has gestureEnable
 set, gWin synthesizes a tap gesture by self sending msgGWinXList.  Thus,
 even though if inputStroke events are turned off in the window, gWin can
 still recognize tap gestures.

 Return Value
	stsInputGrabTerminate		Temporarily grabbing input events
	stsInputTerminate			Not grabbing input events.

 See Also
	msgGWinStroke
	msgGWinKey
	msgGWinForwardKey
	msgGWinBadKey
*/

/****************************************************************************
 msgQuickHelpHelpShow 	takes P_XY32, returns STATUS
	Defined in qhelp.h.

 The theQuickHelpManager sends msgQuickHelpHelpShow to a gWin to ask the 
 gWin to display the gWin's quick help.  (This is the message that
 theQuickHelpManager sends when the user taps while in quick help mode.)

 gWin's default response is to self send msgGWinGesture;  the gesture sent
 along with this msgGWinGesture is a synthesized help gesture.

 If the response to the msgGWinGesture is stsRequestForward, gWin self
 sends msgGWinForwardGesture.  If the response to the msgGWinForwardGesture
 is stsRequestForward, gWin self sends msgQuickHelpShow to
 theQuickHelpManager with a helpId of hlpQuickHelpNoHelp.  (In response to
 this, theQuickHelpManager displays the "no help available" text to the
 user.)
*/

/****************************************************************************
 msgXlateCompleted	takes nothing, returns STATUS
	Defined in xlate.h.

 A gWin's gesture translator sends msgXlateCompleted to the gWin when a
 gesture translation is complete.  (The gWin has previously started the
 translation by sending msgScrComplete to the gesture 
 translator.)  
 
 gWin's default response is to extract the xlist from the translator and
 self send msgGWinXList.

 See Also
 	msgGWinXList
*/

#endif
