/****************************************************************************
 File: geo.h

 (C) Copyright 1992 by Go Corporation, All Rights Reserved.

 $Revision:   1.6  $ 
   $Author:   sbartlet  $
     $Date:   24 Feb 1992 18:25:38  $

 This file contains the API definition for PenPoint's geometry package. The
 package provides points, rectangles, matrices, etc., and is used by the
 graphics and windowing software.

 Typical application software will only need the types defined in this file
 and not need to use the functions.

 The functions described in this file are contained in WIN.LIB. 
****************************************************************************/
#ifndef GEO_INCLUDED
#define GEO_INCLUDED

#ifndef GO_INCLUDED
#include <go.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Typedefs, #defines, and Status Values                                  *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef S32 COORD32;
typedef S16 COORD16;
typedef S16 ANGLE;           // Foley/VanDam counter clockwise angles

typedef struct
{
  FIXED    x,
           y;

} SCALE, * P_SCALE;

typedef struct
{
  COORD32  x,
           y;

} XY32, * P_XY32;

typedef struct
{
  COORD32  w,
           h;

} SIZE32, * P_SIZE32;

typedef struct
{
  XY32    origin;
  SIZE32  size;

} RECT32, * P_RECT32;

typedef struct
{
  COORD16  x,
           y;

} XY16, * P_XY16;

typedef struct
{
  COORD16  w,
           h;

} SIZE16, * P_SIZE16;

typedef struct
{
  XY16    origin;
  SIZE16  size;

} RECT16, * P_RECT16;

/*
 Type MAT represents a 3x3 matrix;  however m13, m23 and m33 are constant 
 and so they are not stored.
//{
    m11    m12    m13
    m21    m22    m23
    m31    m32    m33

    sX     a      0
    a      sY     0
    tX     tY     1
//}
*/

typedef struct
{
   FIXED m11,    
         m12,
         m21,    
         m22;    
   S32   m31,    
         m32;    

} MAT, * P_MAT;

Enum16(GEO_MAT_MULT) {geoPreMultiply,geoPostMultiply};

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Handy macros                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define Coord32To16(c) ((c>0)?(COORD16)Min(c,maxS16):(COORD16)Max(c,minS16))
#define Coord16from32(c) Coord32To16(c)

#define	RectInit(r, _x, _y, _w, _h)	\
	{ (r)->origin.x = (_x); (r)->origin.y = (_y); \
	(r)->size.w = (_w); (r)->size.h = (_h);	}

#define	RectRight(r)	((r)->origin.x + (r)->size.w)
#define	RectTop(r)		((r)->origin.y + (r)->size.h)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Functions	                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 Rect16To32 returns nothing
    Take a RECT16 and produce a RECT32.
*/
void EXPORTED Rect16To32 (
    P_RECT32 p32,  // Out 
    P_RECT16 p16   // In
);

/****************************************************************************
 Rect32To16 returns nothing
    Take a RECT32 and produce a RECT16 with rounding.

 Each 32-bit number is rounded to 16-bits using Coord32To16.
*/
void EXPORTED Rect32To16 (
    P_RECT16 p16,  // Out 
    P_RECT32 p32   // In
);

/****************************************************************************
 Rect32Intersect returns BOOLEAN
    Take two RECT32's and produce their intersection.

 Returns whether the two rectangles intersect.  When TRUE, the rectangle
 returned will always have positive width and height, even though
 either of the parameter rectangles may have negative width or height.
*/
BOOLEAN EXPORTED Rect32Intersect (
    P_RECT32 pA,    // In
    P_RECT32 pB,    // In
    P_RECT32 pRet   // Out: the intersection
);

/****************************************************************************
 Rect32sIntersect returns BOOLEAN
    Test if two RECT32's intersect.

 Either of the parameter rectangles may have negative width or height.
*/
BOOLEAN EXPORTED Rect32sIntersect (
    P_RECT32 pA,    // In
    P_RECT32 pB     // In
);

/****************************************************************************
 Rect32EnclosesRect32 returns BOOLEAN
    Test if a RECT32 encloses another RECT32.

 Returns true if rect A completely encloses rect B.
 Either of the parameter rectangles may have negative width or height.
*/
BOOLEAN EXPORTED Rect32EnclosesRect32 (
    P_RECT32 pA,    // In
    P_RECT32 pB     // In
);

/****************************************************************************
 Rect16Intersect returns BOOLEAN
    Take two RECT16's and produce their intersection.

 Returns whether the two rectangles intersect.  When TRUE, the rectangle
 returned will always have positive width and height, even though
 either of the parameter rectangles may have negative width or height.
*/
BOOLEAN EXPORTED Rect16Intersect (
    P_RECT16 pA,    // In
    P_RECT16 pB,    // In
    P_RECT16 pRet   // Out: the intersection
);

/****************************************************************************
 XY32inRect32 returns BOOLEAN
    Test if an XY32 point is inside a RECT32.
*/
BOOLEAN EXPORTED XY32inRect32 (
    P_RECT32 pRect, // In 
    P_XY32   pXY    // In
);

/****************************************************************************
 Rect32Empty returns BOOLEAN
    Test if a RECT32 has a width or height that is zero.

 Also, if pRect is pNull then this function returns true.
*/
BOOLEAN EXPORTED Rect32Empty (
    P_RECT32 pRect  // In
);

/****************************************************************************
 Rect16Empty returns BOOLEAN
    Test if a RECT16 has a width or height that is zero.

 Also, if pRect is pNull then this function returns true.
*/
BOOLEAN EXPORTED Rect16Empty (
    P_RECT16 pRect  // In
);

/****************************************************************************
 MatCreate returns nothing
    Create a MAT given a translate, rotate, and scale.

 pMat is set to identity. Then the three transformation are post-multiplied
 in the order (1) translate, (2) rotate, and (3) scale.
*/
void EXPORTED MatCreate (
    P_MAT   pMat,     // Out
    COORD32 tX,       // In
    COORD32 tY,       // In
    ANGLE   angle,    // In
    FIXED   sX,       // In
    FIXED   sY        // In
);

/****************************************************************************
 MatIdentity returns nothing
    Set a MAT to the identity matrix.
*/
void EXPORTED MatIdentity (
    P_MAT  // Out
);

/****************************************************************************
 MatRotate returns nothing
    Rotate a MAT.
*/
void EXPORTED MatRotate (
    GEO_MAT_MULT	order,   // In: {geoPreMultiply,geoPostMultiply} 
    P_MAT     		pMat,    // In-Out:
    ANGLE     		angle    // In: 0-359 degrees
);

/****************************************************************************
 MatTranslate returns nothing
    Translate a MAT.
*/
void EXPORTED MatTranslate (
    GEO_MAT_MULT	order,   // In: {geoPreMultiply,geoPostMultiply} 
    P_MAT     		pMat,    // In-Out:
    P_XY32    		xy       // In:
);

/****************************************************************************
 MatScale returns nothing
    Scale a MAT.
*/
void EXPORTED MatScale (
    GEO_MAT_MULT	order,   // In: {geoPreMultiply,geoPostMultiply} 
    P_MAT     		pMat,    // In-Out:
    P_SCALE   		scale    // In:
);

/****************************************************************************
 MatInvert returns nothing
    Invert a MAT.

 pSource is inverted and placed in pDest. pSource and pDest can be
 the same matrix.
*/
void EXPORTED MatInvert (
    P_MAT     pDest,   // Out:
    P_MAT     pSource  // In:
);

/****************************************************************************
 MatMultiply returns nothing
    Multiply two MAT's.

 If order is geoPreMultiply, then answer = right * left.  If order is
 geoPostMultiply, then answer = left * right;
*/
void EXPORTED MatMultiply (
    GEO_MAT_MULT	order,   // In: {geoPreMultiply,geoPostMultiply} 
    P_MAT     		answer,  // Out
    P_MAT     		left,    // In
    P_MAT     		right    // In
);

/****************************************************************************
 MatXYTransform16 returns nothing
    Transform a XY32 producing a XY16 result.

 Each 32-bit number is rounded to 16-bits using Coord32To16.
*/
void EXPORTED MatXYTransform16 (
    P_MAT    pMat,     // In
    P_XY32   pSource,  // In
    P_XY16   pDest     // Out
);

/****************************************************************************
 MatXYTransform32 returns nothing
    Transform a XY32 producing a XY32 result.
*/
void EXPORTED MatXYTransform32 (
    P_MAT    pMat,     // In
    P_XY32   pSource,  // In
    P_XY32   pDest     // Out
);

/****************************************************************************
 MatWHTransform16 returns nothing
    Transform a SIZE32 producing a SIZE16 result.

 This transformation is similar to MatXYTransform16 except the
 translation components of the matrix are ignored and the values
 returned are always positive.

 Each 32-bit number is rounded to 16-bits using Coord32To16.
*/
void EXPORTED MatWHTransform16 (
    P_MAT      pMat,     // In
    P_SIZE32   pSource,  // In
    P_SIZE16   pDest     // Out
);

/****************************************************************************
 MatWHTransform32 returns nothing
    Transform a SIZE32 producing a SIZE32 result.

 This transformation is similar to MatXYTransform32 except the
 translation components of the matrix are ignored and the values
 returned are always positive.
*/
void EXPORTED MatWHTransform32 (
    P_MAT      pMat,     // In
    P_SIZE32   pSource,  // In
    P_SIZE32   pDest     // Out
);

/****************************************************************************
 MatTransformRECT32 returns nothing
    Transform a RECT32.
*/
void EXPORTED MatTransformRECT32 (
    P_MAT      pMat,     // In
    P_RECT32   pSource   // In-Out
);

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Debugging Functions													   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 MatDump	returns nothing
	Prints the fields of a matrix.

 This function may not work unless the debugging version of win.dll is
 being used.
*/
void    EXPORTED MatDump(P_MAT pm);

/****************************************************************************
 DumpRect	returns nothing
	Prints the fields of a rectangle.

 This function may not work unless the debugging version of win.dll is
 being used.
*/
void    EXPORTED DumpRect(P_RECT32 pRect);


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Special Functions													   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/*
 WARNING:  The functions in this section (MatXTransform16, MatYTransform16,
 MatWTransform16, and MatHTransform16) work only in a limited set of cases:
 NO translation, NO rotation, and they perform NO rounding and thus can
 overflow the 16 bit result.

 These functions should not normally be used by application software.
*/
COORD16 EXPORTED MatXTransform16(P_MAT pi, COORD16 x);
COORD16 EXPORTED MatYTransform16(P_MAT pi, COORD16 y);
COORD16 EXPORTED MatWTransform16(P_MAT pi, COORD16 w);
COORD16 EXPORTED MatHTransform16(P_MAT pi, COORD16 h);

#endif


