/****************************************************************************
 File: embedwin.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.24  $
     $Date:   28 Feb 1992 15:38:30  $

 This file contains the API definition for clsEmbeddedWin.

 clsEmbeddedWin inherits from clsGWin.

 Embedded windows provide default functionality for embedding windows,
 move/copy, selection ownership and input target interaction.
****************************************************************************/

/**** Other Important Files ****/
/*
 ewnew.h contains the API definition for creating embeddedWins.
 Of particular interest there are definitions for:
    -:  embedded window style (EMBEDDED_WIN_STYLE)
    -:  embedded window metrics (EMBEDDED_WIN_METRICS)
    -:  new structs (EMBEDDED_WIN_NEW_ONLY, EMBEDDED_WIN_NEW)
    -:  selection types (ewSelect...)
    -:  move/copy modes
*/


/**** Road Map ****/
/*
 Typical subclasses self send:
	-:	msgEmbeddedWinBeginMove
	-:	msgEmbeddedWinBeginCopy
	-:	several messages defined in xfer.h and sel.h

 Typical subclasses handle:
	-:	several messages defined in xfer.h and sel.h

 Subclasses that support traversal (see mark.h) probably handle:
	-:	msgEmbeddedWinShowChild

 Subclasses that manage child windows as part of their data (e.g. text
 editors) probably handle:
	-:	msgEmbeddedWinInsertChild
	-:	msgEmbeddedWinExtractChild
	-:	msgEmbeddedWinPositionChild

 Subclasses that file information other than instance data (e.g. reference
 buttons) probably handle:
	-:	msgEmbeddedWinDestroy
	-:	msgEmbeddedWinGetDest
	-:	msgEmbeddedWinForwardedGetDest

 Subclasses that implement sophisticated printing behavior probably handle:
	-:	msgEmbeddedWinGetPrintInfo
*/


/**** Embedding ****/
/*
 When an embeddedWin has style.embeddor true, it can embed all embeddedWins
 with style.embeddee true.  It can also have embeddedWins moved or copied
 into it.  Examples of embeddors are (1) cork margins, (2) bookshelves, and
 (3) the main window of most applications.  An embeddedWin with
 style.embeddor true also responds to the "link" gesture (xgsDblCircle in
 xgesture.h) by creating a goto button in the window.

 When an embeddedWin has style.embeddee true, the embeddedWin can be
 embedded, moved and copied.  Examples of embeddees are (1) icons for an
 application (2) appWins around an application's frame (see appwin.h) and
 (3) goto buttons (see goto.h).
*/


/**** Move/Copy Behavior ****/
/*
 The header files sel.h and xfer.h describe PenPoint's move/copy mechanism.
 You need to understand PenPoint's general move/copy mechanism before
 you'll be able to understand embeddedWin's specific use of it.

 clsEmbeddedWin defines a data transfer type, xferEmbeddedWin, and a
 corresponding data transfer protocol.  These can be used to move and copy
 embeddedWins.

 Unlike most PenPoint data transfer protocols, the xferEmbeddedWin protocol
 is primarily a "push" protocol -- the destination sends a message to the
 source instructing the source to move/copy itself into the destination.

 If the source and destination agree to move data using xferEmbeddedWin,
 the following steps are taken.  (This discussion assumes that the
 destination's style.quickMove is true;  see section "Move Optimizations"
 for more information.)

    -:  The destination embeddedWin sends msgEmbeddedWinMove to the source
        embeddedWin to have the source move itself into the destination at
        pArgs->xy.

    -:  In response, the source self sends msgEmbeddedWinMoveCopyOK.  If
        the resulting moveOK is false, the source returns
        stsEWSelRefusedMove.
 
    -:  If the destination's parent window is the same as self's parent
        window, then the source embeddedWin "moves" itself by sending
        msgEmbeddedWinPositionChild to the destination. 
 
    -:  If self and the destination are in the same process, then the
        source embeddedWin "moves" itself by sending
        msgEmbeddedWinExtractChild to its parent, and then sending
        msgEmbeddedWinInsertChild to the destination. 

    -:  If self and the destination are in different processes, then the
        source embeddedWin "moves" itself by (1) using msgCopy to create a
        copy of itself that is owned by the destination's process, and (2)
        sending msgEmbeddedWinInsertChild to the destination. Finally the 
        original source embeddedWin posts msgEmbeddedWinDestroy to itself.

 Copying data goes through the following steps:

    -:  The destination embeddedWin sends msgEmbeddedWinCopy to the source
        embeddedWin to have the source copy itself into the destination at
        pArgs->xy.

    -:  In response, the source self sends msgEmbeddedWinMoveCopyOK.  If
        the resulting copyOK is false, the source returns
        stsEWSelRefusedCopy.

    -:  The source embeddedWin "copies" itself by (1) using msgCopy to
        create a copy of itself that is owned by the destination's process,
        and (2) sending msgEmbeddedWinInsertChild to the destination.
*/


/**** Selection Interaction ****/
/*
 The header file sel.h describes PenPoint's selection mechanism.  You need
 to understand PenPoint's general selection mechanism before you can
 understand embeddedWin's specific use of it.

 clsEmbeddedWin provides default selection management for itself and its
 subclasses.

 Some objects should take selection ownership via msgSelSetOwner and some
 should take ownership via msgSelSetOwnerPreserve.  (See sel.h for complete
 information, but here's one example: objects in pop-up dialog boxes, such
 as option sheets, should typically take ownership via
 msgSelSetOwnerPreserve.)

 Rather than having each subclass or instance compute which way to take the
 selection, embeddedWin creators can given an embeddedWin a style.selection
 value which tells the embeddedWin which message to use to take selection
 ownership.

 Subclasses of clsEmbeddedWin should self send msgSelSelect to take
 selection ownership rather than sending msgSelSetOwner or
 msgSelSetOwnerPreserve directly to theSelectionManager.

 In response to msgSelSelect, an embeddedWin does the following:

    -:  If style.selection is ewSelect, the embeddedWin sends
        msgSelSetOwner to theSelectionManager with self as the value of
        pArgs.
 
    -:  If style.selection is ewSelectPreserve, the embeddedWin sends
        msgSelSetOwnerPreserve to theSelectionManager with self as the
        value of pArgs.

    -:  If style.selection is ewSelectUknown (the default), the embeddedWin
        searches up the window hierarchy looking for the first window that
        (1) is an embeddedWin and (2) has a style.selection other than
        ewSelectUnknown.  The value of that window's style.selection is
        used.  If no ancestor sets this bit, or no ancestor is an
        embeddedWin, the embeddedWin takes the selection via
        msgSelSetOwner.

 In addition to selection ownership message, an embeddedWin provides 	
 default responses to several other messages defined in sel.h. Details of
 each response are described with the specific messages later in this file.
*/


/**** Input Target Interaction ****/
/*
 One of PenPoint's UI guidelines is that, in most cases, the selection
 owner should also be the input target.  The input target receives keyboard
 events from the input system.  (See sel.h and input.h for more
 information.)

 While PenPoint as a whole does not enforce a link between selection
 ownership and the input target, clsEmbeddedWin does.  As part of its
 response to msgSelSelect and msgSelPromote, an embeddedWin makes itself
 the input target.
*/

/**** Enabling Move/Copy of the Entire Window ****/
/*
 If you want an entire embeddedWin to be moveable or copyable as a window,
 then you should set style.moveable and style.copyable to true.  Also, you
 should turn on the inputHoldTimeout flag of the window's input flags.
//{
	pArgs->win.flags.input	|= inputHoldTimeout;
//}
*/


/**** Move Optimizations ****/
/*
 By default, an embeddedWin's style.quickMove is true, and the section
 "Move/Copy Behavior" correctly describes what happens during a move.  But
 a client or subclass can set style.quickMove false, and thereby defeat the
 "same parent" and "same process" optimizations.
*/

#ifndef EMBEDWIN_INCLUDED
#define EMBEDWIN_INCLUDED

#ifndef EMBEDWIN_NEW_INCLUDED
#include <ewnew.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

#ifndef PRINT_INCLUDED
#include <print.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT EMBEDDED_WIN, *P_EMBEDDED_WIN;

/**** Status Codes ****/
#define stsEWNoSelection				MakeStatus(clsEmbeddedWin, 1)
#define stsEWSelRefusedMove				MakeStatus(clsEmbeddedWin, 2)
#define stsEWSelRefusedCopy				MakeStatus(clsEmbeddedWin, 3)
#define stsEWSelRefusedLink				MakeStatus(clsEmbeddedWin, 4)
#define stsEWUnrecognizedFormat			MakeStatus(clsEmbeddedWin, 5)
#define stsEWMoveToInvalidLocation		MakeStatus(clsEmbeddedWin, 6)
#define stsEWCopyToInvalidLocation		MakeStatus(clsEmbeddedWin, 7)
#define stsEWNotEmbeddee				MakeStatus(clsEmbeddedWin, 8)
#define stsEWRefusedDelete				MakeStatus(clsEmbeddedWin, 9)


/*
 * xferEmbeddedWin is the data transfer type clsEmbeddedWin uses to move or
 * copy embeddedWins.
*/
#define xferEmbeddedWin					MakeTag(clsEmbeddedWin, 1)




/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Messages								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgEmbeddedWinGetMetrics	takes P_EMBEDDED_WIN_METRICS, returns STATUS
	Passes back an embeddedWin's metrics.

 pArgs->uuid is set if and only if style.embeddee is true.

 See ewnew.h for the definition of P_EMBEDDED_WIN_METRICS.
*/
#define msgEmbeddedWinGetMetrics		MakeMsg(clsEmbeddedWin, 1)


/****************************************************************************
 msgEmbeddedWinGetStyle	takes P_EMBEDDED_WIN_STYLE, returns STATUS
	Passes back an embeddedWin's style.

 See ewnew.h for the definition of P_EMBEDDED_WIN_STYLE.
*/
#define msgEmbeddedWinGetStyle			MakeMsg(clsEmbeddedWin, 2)


/****************************************************************************
 msgEmbeddedWinSetStyle	takes P_EMBEDDED_WIN_STYLE, returns STATUS
	Specifies an embeddedWin's style.

 If pArgs->embeddee is true and the embeddedWin's uuid is nil, a uuid is
 created for the window.  Clients must not alter the value of
 style.moveCopyMode.

 See ewnew.h for the definition of P_EMBEDDED_WIN_STYLE.
*/
#define msgEmbeddedWinSetStyle			MakeMsg(clsEmbeddedWin, 3)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Move/Copy Protocol Messages						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgEmbeddedWinBeginMove	takes P_EMBEDDED_WIN_BEGIN_MOVE_COPY, returns STATUS
	Places an embeddedWin in move mode.

 An embeddedWin self sends this message to get itself into move mode.
 This message is usually self sent by an embeddedWin as part of the
 response to msgSelBeginMove if style.moveable is set.

 clsEmbeddedWin responds by creating a move icon (an instance of 
 clsMoveCopyIcon).  If pArgs->bounds is a visible rectangle, the move icon
 is created with an image of what's displayed in the pArgs->bounds
 rectangle in the embeddedWin.  Otherwise a default move icon is displayed
 centered at pArgs->xy.  The client of the icon is self.  Also
 style.moveCopyMode becomes ewMoveMode.

 Return Value
	stsRequestDenied:	The window is already in either ewMoveMode or
						ewCopyMode

 See Also
	msgSelBeginMove
*/
#define msgEmbeddedWinBeginMove			MakeMsg(clsEmbeddedWin, 4)

typedef struct EMBEDDED_WIN_BEGIN_MOVE_COPY {

	XY32		xy;				// x,y in source to begin move/copy.
	RECT32		bounds;			// Bounding box of area to move/copy.
	U32			reserved[4];	// Reserved.

} EMBEDDED_WIN_BEGIN_MOVE_COPY, *P_EMBEDDED_WIN_BEGIN_MOVE_COPY;


/****************************************************************************
 msgEmbeddedWinBeginCopy	takes P_EMBEDDED_WIN_BEGIN_MOVE_COPY, returns STATUS
	Places an embeddedWin in copy mode.

 An embeddedWin self sends this message to get itself into copy mode.
 This message is usually self sent by an embeddedWin as part of the
 response to msgSelBeginCopy if style.copyable is set. 

 clsEmbeddedWin responds by creating a copy icon (an instance of 
 clsMoveCopyIcon).  If pArgs->bounds is a visible rectangle, the copy icon
 is created with an image of what's displayed in the pArgs->bounds
 rectangle in the embeddedWin.  Otherwise a default copy icon is displayed
 centered at pArgs->xy.  The client of the icon is self.  Also
 style.moveCopyMode becomes ewCopyMode.

 Return Value
	stsRequestDenied:	The window is already in either ewMoveMode or
						ewCopyMode.

 See Also
	msgSelBeginCopy
*/
#define msgEmbeddedWinBeginCopy			MakeMsg(clsEmbeddedWin, 5)


/****************************************************************************
 msgEmbeddedWinMove	takes P_EMBEDDED_WIN_MOVE_COPY, returns STATUS
	Moves an embeddedWin to the destination.

 A destination embeddedWin sends this message to a source embeddedWin
 to have the source embeddedWin move itself to the destination.

 See the section "Move/Copy Behavior" for more information.

 Return Value
	stsEWSelRefusedMove:			The send of msgEmbeddedWinMoveCopyOK 
									returned FALSE for moveOK.
	stsEWMoveToInvalidLocation:		window could not be moved to pArgs->dest.
*/
#define msgEmbeddedWinMove				MakeMsg(clsEmbeddedWin, 6)

typedef struct EMBEDDED_WIN_MOVE_COPY {

	XY32		xy;				// x,y location in dest.
	OBJECT		dest;			// Destination object.
	TAG			format;			// Data transfer format.  Must be
								// xferEmbeddedWin.
	OBJECT		uid;			// out: moved/copied object.
	U32			reserved[2];	// Reserved.

} EMBEDDED_WIN_MOVE_COPY, *P_EMBEDDED_WIN_MOVE_COPY;


/****************************************************************************
 msgEmbeddedWinProvideIcon	takes P_EMBEDDED_WIN_PROVIDE_ICON, returns STATUS
	Asks an embeddedWin to provide the move/copy icon.

 An embeddedWin's default response is as follows:
    -:  if pArgs->bounds.size.w and pArgs->bounds.size.h are both greater
        than zero, then a marquee style icon is created using a "snapshot" 
		of the screen image contained in pArgs->bounds.
    -:  Otherwise, a default move or copy icon is created.
*/
#define msgEmbeddedWinProvideIcon		MakeMsg(clsEmbeddedWin, 23)

typedef struct EMBEDDED_WIN_PROVIDE_ICON {

	MESSAGE		msg;			// msgEmbeddedWinMove or msgEmbeddedWinCopy.
	XY32		xy;				// x,y in source to begin move/copy.
	RECT32		bounds;			// Bounding box of area to move/copy.
	OBJECT		icon;			// out: the icon.
	U32			reserved[4];	// Reserved.

} EMBEDDED_WIN_PROVIDE_ICON, *P_EMBEDDED_WIN_PROVIDE_ICON;


/****************************************************************************
 msgEmbeddedWinCopy		takes P_EMBEDDED_WIN_MOVE_COPY, returns STATUS
	Copies an embeddedWin to the destination.

 A destination embeddedWin sends this message to a source embeddedWin
 to have the source embeddedWin copy itself to the destination.

 See the section "Move/Copy Behavior" for more information.

 Return Value
	stsEWSelRefusedCopy:			The send of msgEmbeddedWinMoveCopyOK 
									returned FALSE for copyOK.
	stsEWCopyToInvalidLocation:		window could not be copied to pArgs->dest.
*/
#define msgEmbeddedWinCopy				MakeMsg(clsEmbeddedWin, 7)


/****************************************************************************
 msgEmbeddedWinMoveCopyOK	takes P_EMBEDDED_WIN_MOVE_COPY_OK, returns STATUS
	Asks whether it is OK to move or copy an embeddedWin to a destination.

 A source embeddedWin self sends this message to check that it is OK to
 move or copy itself to the destination.  The default response to this
 message is to fill in pArgs->moveOK with style.moveable and pArgs->copyOK
 with style.copyable.

 See the section "Move/Copy Behavior" for more information.

 Return Value
	stsEWUnrecognizedFormat:	target.format was not xferEmbeddedWin.
	stsEWNotEmbeddee:			embeddedWin is not an embeddee.

 See Also
	msgEmbeddedWinMove
	msgEmbeddedWinCopy
*/
#define msgEmbeddedWinMoveCopyOK		MakeMsg(clsEmbeddedWin, 8)

typedef struct EMBEDDED_WIN_MOVE_COPY_OK {

	BOOLEAN					moveOK;	// out: true if ok to move.
	BOOLEAN					copyOK;	// out: true if ok to copy.
	EMBEDDED_WIN_MOVE_COPY	target;	// move/copy struct. 

} EMBEDDED_WIN_MOVE_COPY_OK, *P_EMBEDDED_WIN_MOVE_COPY_OK;


/****************************************************************************
 msgEmbeddedWinGetPenOffset	takes P_XY32, returns STATUS
	Passes back the pen offset during move or copy.

 This message allows the destination of a move or copy to determine the 
 actual pen position relative to the lower-left hand corner of the
 move/copy icon.  
 
 When the user lifts the pen, msgSelBeginMove passes the x,y position of
 the icon, not the pen. 
*/
#define msgEmbeddedWinGetPenOffset			MakeMsg(clsEmbeddedWin, 9)


/****************************************************************************
 msgEmbeddedWinGetDest	takes P_EMBEDDED_WIN_GET_DEST, returns STATUS
	Get the destination for embeddedWin move or copy.

 Some source embeddedWins move or copy more than themselves in response to
 msgEmbeddedWinMove or msgEmbeddedWinCopy.  Some also transfer filed
 information.  (For instance, reference buttons have to move filed
 information about the destination of the button.)  The source sends
 msgEmbeddedWinGetDest to the destination to get the file system location
 that the destination wants the source to use for this filed information.
 
 An embeddedWin's default response is to (1) set pArgs->locator to
 OSThisApp()'s locator, (2) set pArgs->sequence to 1, and (3) set
 pArgs->path to the empty string.  Then if style.embedForward is true, 
 msgEmbeddedWinForwardedGetDest is sent to self's parent window.

 Corkboard Windows (clsCorkBoardWin; see cbwin.h) are an example of a class
 that that has a non-default response to this message.  When an embeddedWin
 is copied to a cork margin, it may represent a document, and the source is
 likely to copy not only the window but also the document files to the
 destination.  The cork margin cannot allow the source to copy these files
 into the directory of the cork margin's containing application since then
 the files would look like they're in the parent application -- the wrong
 place!  So in response to msgEmbeddedWinGetDest, a corkboard window
 appends an extra directory level to is ancestor's response to
 msgEmbeddedWinGetDest.
*/
#define msgEmbeddedWinGetDest		MakeMsg(clsEmbeddedWin, 10)

#define ewPropCopyDest				MakeTag(clsEmbeddedWin, 1)	// Private.

typedef struct EMBEDDED_WIN_GET_DEST {

	XY32			xy;						// x,y location in self.
	FS_LOCATOR		locator;				// out: Destination parent app.
	U16				sequence;				// out: Sequence in parent.
	char			path[fsPathBufLength];	// Path buffer for locator.
	OBJECT			source;					// Object to be moved/copied.
	U32				reserved[3];			// Reserved.

} EMBEDDED_WIN_GET_DEST, *P_EMBEDDED_WIN_GET_DEST;


/****************************************************************************
 msgEmbeddedWinForwardedGetDest	takes P_EMBEDDED_WIN_GET_DEST, returns STATUS
	Get the destination for embeddedWin move or copy.

 If a child embeddedWin's style.embedForward is true, then the child sends
 msgEmbeddedWinForwardedGetDest to the parent to allow the parent to
 override all or part of the child's response to msgEmbeddedWinGetDest.

 An embeddedWin's default response to this message is identical to the
 default response to msgEmbeddedWinGetDest.
*/
#define msgEmbeddedWinForwardedGetDest		MakeMsg(clsEmbeddedWin, 22)


/****************************************************************************
 msgEmbeddedWinInsertChild	takes P_EMBEDDED_WIN_INSERT_CHILD, returns STATUS
	Asks an embeddedWin to insert a child window.

 clsEmbeddedWin's default response is as follows;  this is illustrated in
 the sample code below.
    -:  send msgEmbeddedWinGetPenOffset to pArgs->source
    -:  offset pArgs->xy by the value passed back by 
        msgEmbeddedWinGetPenOffset
    -:  send msgWinInsert to pArgs->win with self as the parent.

//{
    XY32        xy;
    WIN_METRICS wm;
	ObjSendUpdateRet(msgEmbeddedWinGetPenOffset, pArgs->source, &xy, 
			SizeOf(xy));
	ObjSendUpdateRet(msgWinGetMetrics, pArgs->win, &wm, SizeOf(wm), s);
	wm.bounds.origin.x = pArgs->xy.x - xy.x;
	wm.bounds.origin.y = pArgs->xy.y - xy.y;
	ObjSendRet(msgWinDelta, pArgs->win, &wm, SizeOf(wm), s);
	wm.options	= wsPosTop;
	wm.parent	= self;
	ObjSendRet(msgWinInsert, pArgs->win, &wm, SizeOf(wm), s);
//}

 This message may be sent during a move/copy operation;  see the section
 "Move/Copy Behavior" for more information.
*/
#define msgEmbeddedWinInsertChild			MakeMsg(clsEmbeddedWin, 11)

typedef struct EMBEDDED_WIN_INSERT_CHILD {

	XY32		xy;				// x,y location in destination.
	OBJECT		win;			// Window to insert/extract/position.
	OBJECT		source;			// Requestor.
	U32			reserved[4];	// Reserved.

} EMBEDDED_WIN_INSERT_CHILD, *P_EMBEDDED_WIN_INSERT_CHILD,
  EMBEDDED_WIN_EXTRACT_CHILD, *P_EMBEDDED_WIN_EXTRACT_CHILD,
  EMBEDDED_WIN_POSITION_CHILD, *P_EMBEDDED_WIN_POSITION_CHILD;


/****************************************************************************
 msgEmbeddedWinExtractChild	takes P_EMBEDDED_WIN_EXTRACT_CHILD, returns STATUS
	Asks an embeddedWin to extract a child window.

 clsEmbeddedWin's default response is to ObjectSend msgWinExtract to
 pArgs->win.
 
 This message may be sent during a move/copy operation;  see the section
 "Move/Copy Behavior" for more information.
*/
#define msgEmbeddedWinExtractChild		MakeMsg(clsEmbeddedWin, 12)


/****************************************************************************
 msgEmbeddedWinPositionChild	takes P_EMBEDDED_WIN_POSITION_CHILD, returns STATUS
	Asks an embeddedWin to reposition a child window.

 clsEmbeddedWin's default response is as follows;  this is illustrated in
 the sample code below.
    -:  send msgEmbeddedWinGetPenOffset to pArgs->source
    -:  offset pArgs->xy by the value passed back by 
        msgEmbeddedWinGetPenOffset
    -:  self send msgWinDelta.

//{
    XY32        xy;
    WIN_METRICS wm;
	ObjSendUpdateRet(msgEmbeddedWinGetPenOffset, pArgs->source, &xy, 
			SizeOf(xy), s);
	ObjSendUpdateRet(msgWinGetMetrics, pArgs->win, &wm, SizeOf(wm), s);
	wm.bounds.origin.x = pArgs->xy.x - xy.x;
	wm.bounds.origin.y = pArgs->xy.y - xy.y;
	ObjSendRet(msgWinDelta, pArgs->win, &wm, SizeOf(wm), s);
//}

 This message may be sent during a move/copy operation;  see the section
 "Move/Copy Behavior" for more information.
*/
#define msgEmbeddedWinPositionChild		MakeMsg(clsEmbeddedWin, 13)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			Linking Related Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgEmbeddedWinShowChild	takes P_EMBEDDED_WIN_SHOW_CHILD, returns STATUS
 	Display a given area of an embeddedWin to the user

 Clients send this message to ask an embeddedWin to show the rectangle
 pArgs->area to the user, scrolling if necessary.

 Note that pArgs->area is relative to pArgs->areaWin.  Therefore handling
 this message may involve transforming pArgs->area to be relative to self.
 This can be accomplished as follows:
//{
	WIN_METRICS	wm;
	wm.bounds = pArgs->area;
	wm.parent = self;
	ObjCallJmp(msgWinTransformBounds, pArgs->areaWin, &wm, s, Error);
//}

 In many cases, subclasses need do nothing;  clsScrollWin's response to
 this message takes care of it all.  
 
 However, if a subclass does its own scrolling, manages embeddees (for
 example, by not having them inserted when off-screen) or uses something
 other than window coordinates to scroll a scroll window, then it needs to
 respond to this message in the following manner:
    -:  ensure that child is inserted and delta'd to the correct place
        (possibly scrolling it into view if needed)
    -:  transform the rect to the child (remember: it may be in some nested
        window)
    -:  scroll as needed to get that rect into view.
    -:  call ancestor.

 clsEmbeddedWin's default response is to set pArgs->child to self, set
 pArgs->childUUID to self's UUID and ObjectSend the message to its parent.
*/
#define msgEmbeddedWinShowChild			MakeMsg(clsEmbeddedWin, 14)

typedef struct EMBEDDED_WIN_SHOW_CHILD {

	WIN			child;				// the child directly below
	UUID		childUUID;			// its UUID
	RECT32		area;				// area to show
	WIN			areaWin;			// window that the area is relative to

} EMBEDDED_WIN_SHOW_CHILD, * P_EMBEDDED_WIN_SHOW_CHILD;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Other Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgEmbeddedWinSetUUID takes P_UUID, returns STATUS
	Specifies an embeddedWin's uuid.

 Gives an embeddedWin a UUID, if style.embeddee is true.
*/
#define msgEmbeddedWinSetUUID			MakeMsg(clsEmbeddedWin, 19)


/****************************************************************************
 msgEmbeddedWinDestroy	takes OBJ_KEY, returns STATUS
	Permanently destroys an embeddedWin.

 This message is sent to an embeddedWin in response to msgSelDelete, or as
 the last step of msgEmbeddedWinMove. This message is different from
 msgDestroy in that this message is sent when the embeddedWin is being
 permanently destroyed and will never be restored. (msgDestroy is sent when
 the embeddedWin is being destroyed but may be restored later.)

 Any subclasses that file data to maintain information as part of their
 embedding behavior should free that data in response to this message.
 They should not free that data in response to msgDestroy.

 clsEmbeddedWin's default response is as follows:
    -:  if style.deleteable is false, return stsEWRefusedDelete.
    -:  Send msgEmbeddedWinDestroy to any child embeddedWins that are in
        the same task.
    -:  Self send msgDestroy.
*/
#define msgEmbeddedWinDestroy			MakeMsg(clsEmbeddedWin, 20)


/****************************************************************************
 msgEmbeddedWinGetPrintInfo	takes P_EMBEDDEE_PRINT_INFO, returns STATUS
	Passes back an embeddedWin's print information.

 This message gives subclasses an opportunity to support more advanced
 printing of embeddedWins.

 clsEmbeddedWin's default response is to set all fields in *pArgs to 0.
*/
#define msgEmbeddedWinGetPrintInfo		MakeMsg(clsEmbeddedWin, 21)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Messages Defined in clsmgr.h					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgFree		takes OBJ_KEY, returns STATUS
	Defined in clsmgr.h.
*/


/****************************************************************************
 msgSave	takes P_OBJ_SAVE, returns STATUS
	Defined in clsmgr.h.

 clsEmbeddedWin saves the embeddedWin's style and UUID.
*/


/****************************************************************************
 msgRestore		takes P_OBJ_RESTORE, returns STATUS
	Defined in clsmgr.h.

 clsEmbeddedWin restores the embeddedWin's style and UUID.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Messages Defined in xfer.h and sel.h				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgXferList	takes OBJECT, returns STATUS
	Defined in xfer.h.

 This message is sent to an object to ask it to provide the list of data
 transfer types it can provide.

 clsEmbeddedWin's default response is to add the transfer type 
 xferEmbeddedWin to the end of the list.
*/


/****************************************************************************
 msgSelMoveSelection		takes P_XY32, returns STATUS
	Defined in sel.h.

 This message is sent to an object to ask it to move the selection to
 itself.

 See the section "Move/Copy Behavior" for more information.

 Return Value
	stsRequestForward:		embeddedWin is not an embeddor.
	stsEWSelRefusedMove:	destination embeddedWin refused the move.
	stsEWNoSelection:		No selection exists in the system.
*/


/****************************************************************************
 msgSelCopySelection	takes P_XY32, returns STATUS
	Defined in sel.h.

 This message is sent to an object to ask it to copy the selection to
 itself.

 See the section "Move/Copy Behavior" for more information.

 Return Value
	stsRequestForward:		embeddedWin is not an embeddor.
	stsEWSelRefusedCopy:	destination embeddedWin refused the copy.
	stsEWNoSelection:		No selection exists in the system.
*/


/****************************************************************************
 msgSelRememberSelection		takes P_XY32, returns STATUS
	Defined in sel.h.

 Self sent by an embeddedWin in response to the Circle-Tap gesture.  
 
 clsEmbeddedWin's default response is to
	-:	create a reference button
	-:	insert the button by self sending msgEmbeddedWinInsertChild.
 
 Return Value
	stsRequestForward:		The window is not an embeddor.
	stsEWNoSelection:		No selection exists.
*/


/****************************************************************************
 msgSelSelect	takes nothing, returns STATUS
	Defined in sel.h.

 See the section "Selection Interaction" for a description of an 
 embeddedWin's response to msgSelSelect.
*/


/****************************************************************************
 msgSelPromote	takes nothing, returns STATUS
	Defined in sel.h.

 clsEmbeddedWin's default response is to become the input target by calling
 InputSetTarget (see input.h) with self as the target.
*/


/****************************************************************************
 msgSelYield		takes BOOLEAN, returns STATUS
	Defined in sel.h.

 clsEmbeddedWin's default response is to return stsOK.
*/


/****************************************************************************
 msgSelIsSelected	takes nothing, returns BOOLEAN
	Defined in sel.h.

 Return Value
 	stsOK:		self is the selection owner.
	other:		self is not the selection owner.  (Note that self may
				be the preserved selection owner.)
*/


/****************************************************************************
 msgSelDelete	takes U32, returns STATUS
	Defined in sel.h.

 See sel.h for a complete description of when this message is sent.
 Typically, an embeddedWin receives this message because the destination of
 the move is deleting the source.

 embeddedWin's default response is to self send msgEmbeddedWinDestroy.
*/


/****************************************************************************
 msgSelBeginMove		takes nothing, returns STATUS
	Defined in sel.h.

 See sel.h for a complete description of when this message is sent.

 clsEmbeddedWin's default response is to self send msgEmbeddedWinMove.

 Return Value
	stsRequestDenied:		the embeddedWin is already in move or copy mode.
*/


/****************************************************************************
 msgSelBeginCopy		takes nothing, returns STATUS
	Defined in sel.h.

 See sel.h for a complete description of when this message is sent.

 clsEmbeddedWin's default response is to self send msgEmbeddedWinCopy.

 Return Value
	stsRequestDenied:		the embeddedWin is already in move or copy mode.
*/


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *								Other Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgIconProvideBitmap		takes P_ICON_PROVIDE_BITMAP, returns STATUS
	Defined in icon.h.

 An embeddedWin receives this message from a move/copy icon (since the
 embeddedWin is the icon's client.)

 clsEmbeddedWin's default response is to forward the message to OSThisApp().
*/


/****************************************************************************
 msgMoveCopyIconDone	takes OBJECT, returns STATUS
	Defined in mcicon.h.

 An embeddedWin receives this message when a move/copy icon completes.
 (The move/copy icon completes when it dropped on some destination
 window.)

 clsEmbeddedWin's default response is to send msgSelMoveSelection or
 msgSelCopySelection (as appropriate) to the destination window.
*/


/****************************************************************************
 msgMoveCopyIconCancel	takes OBJECT, returns STATUS
	Defined in mcicon.h.

 An embeddedWin receives this message when a move/copy icon is canceled.
 clsEmbeddedWin's default response is to take itself out of move/copy mode
 (by setting self's style.moveCopyMode to ewMoveCopyModeOff).
*/


/****************************************************************************
 msgTrackProvideMetrics	takes P_TRACK_METRICS, returns STATUS
	Defined in track.h.
 
 An embeddedWin receives this message from the move/copy icon's tracker.
 (The tracker can be recognized as the move/copy icon's tracker because
 pArgs->tag will be tagMoveCopyIconTrack.)

 Subclasses can handle this message by repositioning the tracker (and
 therefore the move/copy icon) relative to the pen.  This is done by
 modifying pArgs->initRect.  Typically you do not call the ancestor in such
 cases.  For instance, PenPoint's text component "jumps" the icon so that
 the pen is at the vertical center of the left edge of the icon by using
 code similar to the following:

//{
	MsgHandlerArgType(SomeViewTrackProvideMetrics, P_TRACK_METRICS)
	{
		if (pArgs->tag == tagMoveCopyIconTrack) {
			pArgs->initRect.origin = pArgs->origXY;
			pArgs->initRect.origin.y -= pArgs->initRect.size.h/2;
			return stsOK;
		} else {
			return ObjectCallAncestorCtx(ctx);
		}
		...
	}
//}
*/

#endif	// EMBEDWIN_INCLUDED
