/****************************************************************************
 bytarray.h

 Copyright 1992, GO Corporation, All Rights Reserved.

 $Revision:   1.17  $
   $Author:   msapsfor  $
     $Date:   27 Feb 1992 12:07:28  $

 This file contains the API definition for the ByteArray interface.
 The functions described in this file are contained in MISC.LIB.

 A ByteArray implements a growing and shrinking array of bytes, 
 indexed from 0 to ByteArrayLength()-1.  A ByteArray grabs and 
 releases memory as needed.

 The ByteArray implementation is optimized for highly localized
 series of insertions and deletions.
****************************************************************************/
#ifndef BYTARRAY_INCLUDED
#define BYTARRAY_INCLUDED $Revision:   1.17  $ 

#ifndef	CLSMGR_INCLUDED
#include <clsmgr.h>
#endif
#ifndef	DEBUG_INCLUDED
#include <debug.h>
#endif
#ifndef OSHEAP_INCLUDED
#include <osheap.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Types and Constants                                                    *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef struct BYTE_ARRAY * P_BYTE_ARRAY;

#define stsBAMaxExceeded MakeStatus(clsMisc, 255)

typedef U32 BYTE_INDEX, * P_BYTE_INDEX;
#define SIZE_OF_BYTE_INDEX 4

#define maxBYTE_INDEX maxU32


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Private                                                                *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

//REFGEN BEGINIGNORE
// 
// The BYTE_ARRAY struct implements the ByteArray abstraction using a
// single contiguous memory buffer (specified by firstPart, with
// allocated length specified by bufferLength and actual usage by
// length).  The buffer contains a moving gap whose beginning is at
// (firstPart + firstPartLength).
// 
// Field secondPart accelerates access to characters beyond the gap; it
// is kept equal to (firstPart + ByteArrayGapLength()) to allow direct
// array access to bytes in the second part without offsetting of the
// byte index, i.e. access is via secondPart[i], not
// secondPart[i-firstPartLength].
// 
// Field mode is really of type OS_HEAP_MODE, but we don't want to
// include all of osheap.h in this file.  The mode controls how memory
// allocations and de-allocations are handled.
// 
//REFGEN ENDIGNORE

typedef struct BYTE_ARRAY {
	BYTE_INDEX	length;				// Number of bytes stored in buffer 
	BYTE_INDEX	bufferLength;		// Number of bytes in buffer 
	P_U8		firstPart;			// Beginning of the buffer 
	BYTE_INDEX 	firstPartLength;	// Number of bytes in first part 
	P_U8		secondPart;			// see comments above 
	U16			mode;				// see comments above
} BYTE_ARRAY;


/****************************************************************************
 ByteArrayGapLength returns BYTE_INDEX
	Returns the size of the byte array's gap.
*/
#define ByteArrayGapLength(p) \
	((p)->bufferLength - (p)->length)


/****************************************************************************
 ByteArrayPrint returns void
	Prints the content of the byte array.
*/
#ifdef DEBUG

void EXPORTED
ByteArrayPrint(
	P_BYTE_ARRAY	p,
	P_STRING		charFmt,
	int				charWidth);

#endif // DEBUG

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *  Exported Functions and Macros                                          *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 ByteArrayFindByte returns P_U8
	Gets address of byte n from ByteArray p.

 Warning 1:  n is evaluated twice, so it should not be an expression
 with an auto-increment or decrement!

 Warning 2:  to be as fast as possible, ByteArrayFindByte does no
 error checking!
*/
#define ByteArrayFindByte(p,n) ( \
		(n) < (p)->firstPartLength \
		? &((p)->firstPart[(n)]) \
		: &((p)->secondPart[(n)]) )

		
/****************************************************************************
 ByteArrayFindIndex returns BYTE_INDEX
	Determines the index from address addr of byte in ByteArray p.

 This is the inverse of ByteArrayFindByte.

 Warnings from ByteArrayFindByte apply here also.
*/
#define ByteArrayFindIndex(p,addr) ( \
		(addr) < &((p)->firstPart[(p)->firstPartLength]) \
		? (BYTE_INDEX)(addr - (p)->firstPart) \
		: (BYTE_INDEX)(addr - (p)->secondPart))


/****************************************************************************
 ByteArrayGetByte returns U8
	Get byte n from ByteArray p

 Warnings from ByteArrayFindByte apply here also.
*/
#define ByteArrayGetByte(p,n) \
		((n) < (p)->firstPartLength \
		? (p)->firstPart[(n)] \
		: (p)->secondPart[(n)])

		
/****************************************************************************
 ByteArrayCreate returns STATUS
	Creates a byte array.
 
 Only the osHeapLocal/osHeapShared flags of mode are meaningful.  The
 initial length doesn't matter very much, since the byte array grows
 or shrinks as needed.  However, if length is approximately correct,
 then early insertions will be quicker.  If length<=0, a length of 1
 is assumed.

 Returns stsOK if able to create the byte array, in which case *pp
 will be the created byte array, otherwise *pp will be
 Nil(P_BYTE_ARRAY).

 The mode parameter is really of type OS_HEAP_MODE.
*/
STATUS EXPORTED
ByteArrayCreate(
	P_BYTE_ARRAY * 	pp,
	U16				mode,
	BYTE_INDEX 		length);


/****************************************************************************
 ByteArrayDestroy returns void
	Destroys a byte array.
*/
void EXPORTED 
ByteArrayDestroy(
	P_BYTE_ARRAY	p);


/****************************************************************************
 ByteArrayGetMany returns STATUS
	Gets one or more characters from contiguous positions in the byte array.

 Retrieves up to bufLen characters in p from positions
 [pos..MIN(pos+bufLen,ByteArrayLength(p)).  Client should insure that
 buf != Nil(P_U8).  Returns count of bytes placed in buf.
*/
STATUS EXPORTED
ByteArrayGetMany(
	P_BYTE_ARRAY	p,
	BYTE_INDEX 		pos,
	P_U8			buf,
	BYTE_INDEX 		bufLen);


/****************************************************************************
 ByteArrayReplace returns STATUS
	Replaces zero or more characters in the byte array.

 Replaces len characters in p at positions [pos..pos+len) by bufLen
 characters from buf.  Client should insure that pos+len <=
 ByteArrayLength(p).

 Returns:
	stsOutOfMem: 					if no memory available, or
	stsBadParam:					if range [pos..pos+len) is invalid, or
	stsBAMaxExceeded:				if the maximum ByteArray length
									is exceeded, or
	number bytes taken from buf:	otherwise.
*/
STATUS EXPORTED
ByteArrayReplace(
	P_BYTE_ARRAY	p,
	BYTE_INDEX 		pos,
	BYTE_INDEX 		len,
	P_U8			buf,
	BYTE_INDEX 		bufLen);


/****************************************************************************
 ByteArrayInsert returns STATUS
	Inserts bufLen characters from buf into p at position pos.

 This routine does no error checking.  Client should insure that:
 pos <= ByteArrayLength(p).

 See ByteArrayReplace for possible return values.
*/
#define ByteArrayInsert(p, pos, buf, bufLen)	\
	ByteArrayReplace((p), (pos), 0, (buf), (bufLen))


/****************************************************************************
 ByteArrayDelete returns void
	Delete n characters from p starting at pos.

 This routine does no error checking.  Client should insure that:
 pos+len <= ByteArrayLength(p).
*/
#define ByteArrayDelete(p, pos, len) \
	(void) ByteArrayReplace((p), (pos), (len), Nil(P_U8), 0)


/****************************************************************************
 ByteArrayLength returns BYTE_INDEX
	Returns the number of bytes currently stored in the BYTE_ARRAY.
*/
#define ByteArrayLength(p) ((p)->length)


/****************************************************************************
 ByteArrayHeapMode returns OS_HEAP_MODE
	Returns the heap mode the BYTE_ARRAY was created with.
*/
#define ByteArrayHeapMode(p)	((p)->mode)


/****************************************************************************
 ByteArrayReserve returns STATUS
	Reserves space in byte array (without actually initializing it).

 Reserves len characters in p at position pos, but does not
 initialize them.  (The gap is guaranteed to not break the reserved
 range.) Client should insure that pos <= ByteArrayLength(p).

 Returns:
	stsOutOfMem: 		if no memory available, or
	stsBadParam:		if pos is invalid, or
	stsBAMaxExceeded:	if the maximum ByteArray length is exceeded, or
	stsOK:				otherwise.
*/
STATUS EXPORTED
ByteArrayReserve(
	P_BYTE_ARRAY	p,
	BYTE_INDEX 		pos,
	BYTE_INDEX 		len);


/****************************************************************************
 ByteArrayWrite returns STATUS
	Writes the content of the byte array to the specified file.

 The file parameter must act like a FILE_HANDLE object.
*/
STATUS EXPORTED
ByteArrayWrite(
	P_BYTE_ARRAY	p,
	OBJECT			file);


/****************************************************************************
 ByteArrayRead returns STATUS
	Reads previously saved content of a byte array from the specified file.

 The file parameter must act like a FILE_HANDLE object.
*/
STATUS EXPORTED
ByteArrayRead(
	P_BYTE_ARRAY * 	pp,
	OBJECT			file,
	OS_HEAP_MODE	mode);


/****************************************************************************
 BAFileWriteString returns STATUS
	Debugging utility routine to write a string to a file.  

 Useful when initially writing filing code to insert helpful strings
 into the file and to then skip over the strings when reading the
 file.
 
 This routine takes an exception if it encounters an error.  Also, it
 will only work with a string whose length is MAX_STR_LENGTH or less.

 The file parameter must act like a FILE_HANDLE object.
*/
#ifdef DEBUG

STATUS EXPORTED
BAFileWriteString(
	OBJECT		file,
	P_U8		str);

#endif


/****************************************************************************
 BAFileReadString returns STATUS
	Debugging utility routine to read a string from a file.  

 Useful when initially writing filing code to skip over  strings
 written with BAFileWriteString.
 
 This routine takes an exception if it encounters an error.  Also, it
 will only work with a string whose length is MAX_STR_LENGTH or less.

 The file parameter must act like a FILE_HANDLE object.
*/
#ifdef DEBUG

STATUS EXPORTED
BAFileReadString(
	OBJECT		file,
	P_U8		str);

#endif

#endif // BYTARRAY_INCLUDED
