/****************************************************************************
 File: auxnbmgr.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.46  $
   $Author:   sisaac  $
     $Date:   04 Mar 1992 13:53:56  $

 This file contains the class definition and methods for clsAuxNotebookMgr.
 clsAuxNotebookMgr inherits from clsObject.
 Manages the system notebooks and documents on the bookshelf.

 There is a single instance of clsAuxNotebookMgr in the system; the 
 well-known uid theAuxNotebookMgr. 

 The auxiliary notebook manager creates the following items on the bookshelf:

	The Help Notebook 			   
	The Settings Notebook		   
	The Accessories Pallete			   
	The Stationery Notebook		   
	A Keyboard Instance
	A Connections Notebook Instance
	The Inbox Notebook			   
	The Outbox Notebook			   

 It provides access to those items that are guaranteed to always be on
 the bookshelf:

	The Help Notebook 			   
	The Settings Notebook	   
	The Accessories Pallete			   
	The Stationery Notebook		   
	The Inbox Notebook
	The Outbox Notebook			   

 It allows documents and sections to be created in the Notebooks it manages,
 and copies documents into the Notebooks. It also provides several
 Stationery-specific functions.

 theAuxNotebookMgr is usually not used by applications, other than to 
 activate and open one of system items on the bookshelf.  

 The document/section creation and copy facilities are used by application 
 installation. 
****************************************************************************/
#ifndef AUXNBMGR_INCLUDED
#define AUXNBMGR_INCLUDED

#ifndef GEO_INCLUDED
#include <geo.h>
#endif

#ifndef FS_INCLUDED
#include <fs.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  		Common #defines and typedefs  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* 
 Which bookshelf item? Used in most messages to theAuxNotebookMgr. Also 
 used as part of the definition of the well-known uuids for these items.
*/ 
typedef enum ANM_AUX_NOTEBOOK {
	anmReserved					  = 0,	// Never use this value! See 
										// anmAttrWhichAuxNB below. 
	anmSettingsNotebook			  = 1, 	// Settings Notebook.
	anmHelpNotebook				  = 3,	// Help Notebook.
	anmStationeryNotebook		  = 4,	// Stationery Notebook.
	anmInboxNotebook			  = 5,	// Inbox.
	anmOutboxNotebook			  = 6,	// Outbox.
	anmAccessories				  = 7,	// Accessories Pallette.
} ANM_AUX_NOTEBOOK, *P_ANM_AUX_NOTEBOOK;

/* Exist behavior for creating sections and docs. */
typedef enum ANM_EXIST_BEHAVIOR {
	anmExistGenError,
	anmExistDoNothing,
	anmExistTruncate,
	anmExistGenUnique
} ANM_EXIST_BEHAVIOR, *P_ANM_EXIST_BEHAVIOR;

/* Should a section and/or a notebook entry be added to the stationery menu?*/
typedef struct STAT_MENU_STYLE {
	U16	section		   	: 2,	// Add a section entry.
    	notebook	   	: 2,	// Add a notebook entry.
		unused1			: 12;	// reserved
} STAT_MENU_STYLE, *P_STAT_MENU_STYLE;		  

/****  Filesystem Attributes  ****/
/* Should a given piece of stationery be on the stationery menu? */
#define anmAttrStationeryMenu		FSMakeFix32Attr(clsAuxNotebookMgr, 1)
typedef enum ANM_ATTR_STATIONERY_MENU {
	anmNotOnMenu 	  	  = 0,	 // Same as no attribute.
	anmOnMenu			  = 1
} ANM_ATTR_STATIONERY_MENU;

/* 
 Should a stationery or tool document not be loaded at install time? 
 This attribute is on the the document on the external filesystem.   
*/
#define anmAttrNoLoad				FSMakeFix32Attr(clsAuxNotebookMgr, 2)
typedef enum ANM_ATTR_NO_LOAD {
	anmLoad			  	  = 0,	 // Same as no attribute.
	anmNoLoad			  = 1
} ANM_ATTR_NO_LOAD;

/* Id tag; used to designate stationery or accessory documents. */
#define anmAttrId					FSMakeFix32Attr(clsAuxNotebookMgr, 3)

/* 
 Attribute used to tell the difference between an auxiliary notebooks and 
 a data notebooks. Backup programs take note. Never backup an auxilary  
 notebook!         
*/                                                     
#define anmAttrAuxNB				FSMakeFix32Attr(clsAuxNotebookMgr, 4)
typedef enum ANM_ATTR_AUX_NB {
	anmDataNB		  	  = 0,	 // Same as no attribute.
	anmAuxNB			  = 1
} ANM_ATTR_AUX_NB;

/* 
 Attribute used by clsNBToc to perform special behavior for each auxnb.
 This attribute is stamped on the auxnb's TOC at initialization time.	 
 The attribute values are specified in the ANM_AUX_NOTEBOOK enum.		 
 Note: ANM_AUX_NOTEBOOK must never have a 0 value; 0 indicates no		 
 anmAttrWhichAuxNB attribute.                                            
*/
#define anmAttrWhichAuxNB			FSMakeFix32Attr(clsAuxNotebookMgr, 5)

/* Used to get auto-expand behavior of stationery sections. */
#define anmAttrExpandStationerySection	FSMakeFix32Attr(clsAuxNotebookMgr, 6)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * 				  				  	Messages  		  					   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew						takes P_ANM_NEW, returns STATUS
	category: class message
	Creates a new auxiliary notebook manager. 

 Note: this is done once and only once in the init routine of this dll to
 create theAuxNotebookMgr. This message must never be called by anyone
 else!
*/

#define	auxNotebookMgrNewFields				\
	objectNewFields

typedef struct ANM_NEW {
	auxNotebookMgrNewFields
} ANM_NEW, *P_ANM_NEW;


/****************************************************************************
 msgANMCreateDoc			takes P_ANM_CREATE_DOC, returns STATUS
	Create a document in one of the auxiliary notebooks.

*/
#define msgANMCreateDoc						MakeMsg(clsAuxNotebookMgr, 1)

typedef struct ANM_CREATE_DOC {
	ANM_AUX_NOTEBOOK	notebook;	// Which auxiliary notebook?
	CLASS				docClass;	// Document class.
	P_STRING			pPath;		// Path to create doc in, relative to
									// base of the aux notebook. pNull
									// says to create at top level.
	P_STRING			pName;		// Name of doc. 
	U32					sequence;	// Sequence number to create in front of.
	P_STRING			pBookmarkLabel;	// pNull for no bookmark.
	ANM_EXIST_BEHAVIOR	exist;		// What to do if the doc exists/doesn't 
									// exist. Note: doc might exist due to 
									// warm boot.
	BOOLEAN				putInMenu;	// If type is stationery, should the doc 
									// initially be in the stationery menu?
	P_FS_FLAT_LOCATOR	pDestPath;	// Out: Location of created doc. 
									// if pDestPath is pNull then nothing is
									// returned.
	U32					id;			// Id to tag everything with. 0 is no tag.
} ANM_CREATE_DOC, *P_ANM_CREATE_DOC;


/****************************************************************************
 msgANMCreateSect				takes P_ANM_CREATE_SECT, returns STATUS
	Create a section in one of the auxiliary notebooks.

*/
#define msgANMCreateSect					MakeMsg(clsAuxNotebookMgr, 2)

typedef struct ANM_CREATE_SECT {
	ANM_AUX_NOTEBOOK	notebook;	// Which auxiliary notebook?
	CLASS				sectClass;	// Section class.
	P_STRING			pPath;		// Path to create section in, relative to
									// base of the aux notebook. pNull
									// says to create at top level.
	P_STRING			pName;		// Name of section.
	U32					sequence;	// Sequence number to create in front of.
	P_STRING			pBookmarkLabel;	// pNull for no bookmark.
	ANM_EXIST_BEHAVIOR	exist;		// What to do if the sect exists/doesn't 
									// exist. Note: sect might exist due to 
									// warm boot.
	P_FS_FLAT_LOCATOR	pDestPath;	// Out: Location of created section.
									// if pDestPath is pNull then nothing is
									// returned.
	U32					id;			// Id to tag everything with. 0 is no tag.
} ANM_CREATE_SECT, *P_ANM_CREATE_SECT;


/****************************************************************************
 msgANMMoveInDoc			takes P_ANM_MOVE_COPY_DOC, returns STATUS
	Move a document into an auxiliary notebook.

*/
#define msgANMMoveInDoc						MakeMsg(clsAuxNotebookMgr, 3)

typedef struct ANM_MOVE_COPY_DOC {
	ANM_AUX_NOTEBOOK	notebook;	// Which auxiliary notebook?
	FS_LOCATOR			source;		// Source document.
	P_STRING			pPath;		// Path to move/copy doc to, relative to
									// base of the aux notebook. pNull
									// says to create at top level.
	CLASS				defaultClass;// Class to use if source isn't stamped.
	U32					sequence;	// Sequence number to move/copy in front
									// of.
	P_STRING			pBookmarkLabel;	// pNull for no bookmark.
	ANM_EXIST_BEHAVIOR	exist;		// What to do if the doc exists/doesn't 
									// exist. Note: doc might exist due to 
									// warm boot.
	BOOLEAN				forceInMenu;// If this is stationery, override 
									// any local attribute and put it in 
									// the stationery menu.
	P_FS_FLAT_LOCATOR	pDestPath;	// Out: Location of destination doc. 
									// if pDestPath is pNull then nothing is
									// returned.
	U32					id;			// Id to tag everything with. 0 is no tag.
} ANM_MOVE_COPY_DOC, *P_ANM_MOVE_COPY_DOC;


/****************************************************************************
 msgANMCopyInDoc			takes P_ANM_MOVE_COPY_DOC, returns STATUS
	Copy a document into an auxiliary notebook.

*/
#define msgANMCopyInDoc						MakeMsg(clsAuxNotebookMgr, 4)


/****************************************************************************
 msgANMDelete				takes P_ANM_DELETE, returns STATUS
	Delete a section or document in one of the auxiliary notebooks.

*/
#define msgANMDelete						MakeMsg(clsAuxNotebookMgr, 7)

typedef struct ANM_DELETE {
	ANM_AUX_NOTEBOOK	notebook;	// Which auxiliary notebook?
	P_STRING			pPath;		// Path of item to delete.
} ANM_DELETE, *P_ANM_DELETE;


/****************************************************************************
 msgANMDeleteAll			takes P_ANM_DELETE_ALL, returns STATUS
	Delete all the nodes that are identified by 'id'.

 If a node's id attribute or its app class is 'id' then delete it.

*/
#define msgANMDeleteAll						MakeMsg(clsAuxNotebookMgr, 8)

typedef struct ANM_DELETE_ALL {
	ANM_AUX_NOTEBOOK	notebook;	// Which auxiliary notebook?
	U32					id;			// Id.
} ANM_DELETE_ALL, *P_ANM_DELETE_ALL;


/****************************************************************************
 msgANMGetNotebookPath			takes P_ANM_GET_NOTEBOOK_PATH, returns STATUS
	Returns the base path of one of the auxiliary notebooks.

 Note: This will return a path to the table of contents of the notebook. See
 msgANMGetNotebookUUID if you want the actual notebook itself.
*/
#define msgANMGetNotebookPath				MakeMsg(clsAuxNotebookMgr, 9)

typedef struct ANM_GET_NOTEBOOK_PATH {
	ANM_AUX_NOTEBOOK	notebook;	// Which auxiliary notebook?
	P_FS_FLAT_LOCATOR	pLocator;	// Out: base location of notebook.
									// pNull is returned if the 	
									// notebook does not exist yet.
} ANM_GET_NOTEBOOK_PATH, *P_ANM_GET_NOTEBOOK_PATH;


/****************************************************************************
 msgANMGetNotebookUUID		takes P_ANM_GET_NOTEBOOK_UUID, returns STATUS
	Returns the uuid of one of the auxiliary notebooks.

 Note: This is the UUID of the actual notebook. Use msgANMGetNotebookPath
 to get to the table of contents of the notebook.
*/
#define msgANMGetNotebookUUID				MakeMsg(clsAuxNotebookMgr, 10)

typedef struct ANM_GET_NOTEBOOK_UUID {
	ANM_AUX_NOTEBOOK	notebook;	// Which auxiliary notebook?
	UUID				uuid;		// Out: uuid of auxiliary notebook.
} ANM_GET_NOTEBOOK_UUID, *P_ANM_GET_NOTEBOOK_UUID;


/****************************************************************************
 msgANMOpenNotebook			takes P_ANM_OPEN_NOTEBOOK, returns STATUS
	Activate and optionally open an auxiliary notebook.

*/
#define msgANMOpenNotebook					MakeMsg(clsAuxNotebookMgr, 11)

typedef struct ANM_OPEN_NOTEBOOK {
	ANM_AUX_NOTEBOOK			notebook;	  // Which notebook.
	BOOLEAN						activateOnly; // Only activate; don't open
	OBJECT						uid;		  // Out: uid of activated or 
											  // opened auxnb.
} ANM_OPEN_NOTEBOOK, *P_ANM_OPEN_NOTEBOOK;


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *	Private																   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/****************************************************************************
 msgANMPopUpStationeryMenu			takes P_ANM_POP_UP_MENU, returns STATUS
	Pop up the stationery menu at the specified location. 

 If the user hits one of the menu items create a stationery document in the 
 destination object at the hotSpot.
*/
#define msgANMPopUpStationeryMenu 			MakeMsg(clsAuxNotebookMgr, 5)

typedef struct ANM_POP_UP_MENU {
	XY32			    hotSpot;	// Where to pop up menu. Coords are 
									// relative to destObj.
	OBJECT				destObj; 	// Object to create stationery in front 
									// of.
	STAT_MENU_STYLE		style;		// Menu style.
} ANM_POP_UP_MENU, *P_ANM_POP_UP_MENU;


/****************************************************************************
 msgANMGetStationeryMenu			takes P_ANM_GET_MENU, returns STATUS
	Passes back the stationery menu. 

 This message allows the app framework to add the stationery menu to
 an existing menu bar. When the stationery menu is invoked, stationery
 is created in destObj at the hotSpot.
*/
#define msgANMGetStationeryMenu 			MakeMsg(clsAuxNotebookMgr, 6)

typedef struct ANM_GET_MENU {
	XY32			    hotSpot;	// Where to pop up menu. Coords are 
									// relative to destObj.
	OBJECT				destObj; 	// Object to create stationery in front 
									// of.
	STAT_MENU_STYLE		style;		// Menu style.
	OBJECT				menu;		// Out: Stationery menu.
} ANM_GET_MENU, *P_ANM_GET_MENU;


/****************************************************************************
 msgANMAddToStationeryMenu	takes P_ANM_MENU_ADD_REMOVE, returns STATUS
	Add a stationery notebook doc to the stationery menu. 

*/
#define msgANMAddToStationeryMenu			MakeMsg(clsAuxNotebookMgr, 12)

typedef struct ANM_MENU_ADD_REMOVE {
	UUID						document;  // Dir Index of document to remove.
} ANM_MENU_ADD_REMOVE, *P_ANM_MENU_ADD_REMOVE;


/****************************************************************************
 msgANMRemoveFromStationeryMenu	takes P_ANM_MENU_ADD_REMOVE, returns STATUS
	Remove a document from the stationery menu 

*/
#define msgANMRemoveFromStationeryMenu		MakeMsg(clsAuxNotebookMgr, 13)


/****************************************************************************
 msgANMStationeryMenuNameChanged takes P_ANM_MENU_NAME_CHANGED, returns STATUS
	Informs the stationery menu that one of its documents has changed name.

*/
#define msgANMStationeryMenuNameChanged		MakeMsg(clsAuxNotebookMgr, 17)

typedef struct ANM_MENU_NAME_CHANGED {
	UUID						document;  // Dir Index of document whose name
										   // changed.
} ANM_MENU_NAME_CHANGED, *P_ANM_MENU_NAME_CHANGED;

/* Obsolete */
#define anmAttrPermanent  			FSMakeFix32Attr(clsAuxNotebookMgr, 0)
typedef enum ANM_ATTR_PERMANENT {
	anmNotPermanent 	  = 0,	 // Same as no attribute.
	anmPermanent		  = 1
} ANM_ATTR_PERMANENT;

// Next available messsage number: 18

#endif	// AUXNBMGR_INCLUDED


