/****************************************************************************
 File: appmgr.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.30  $
   $Author:   pstahl  $
     $Date:   19 Mar 1992 12:24:16  $

 This file contains the API definition for clsAppMgr.

 clsAppMgr inherits from clsClass.
 Provides support for application classes and document management.

 "AppMgr" stands for Application Manager.
****************************************************************************/

/**** Introduction ****/
/*
 When you create a new application class (i.e., install an application), 
 rather than sending msgNew to clsClass you send msgNew to clsAppMgr.
 This allows you to specify properties of the application class, 
 and also to specify in advance some default properties of the documents
 (i.e., instances) of the application class.

 There is one instance of clsAppMgr for each installed application class.
 This object is given the well-known uid of the application class.
 The application manager class implements document management messages and 
 stores information about the installed application class in its instance
 data.
*/

#ifndef APPMGR_INCLUDED
#define APPMGR_INCLUDED

#include <fs.h>
#include <geo.h>

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT APPMGR, *P_APPMGR;

/****  AppMgr Flags  ****/									  
/*
 Various settings for the installed application class.

 stationery:  If true, an instance of the application will be placed
 in the Stationery Notebook when the application is installed.  The
 instance will have default parameters.  You can also create customized
 stationery instances using the STATNRY subdirectory.  See appmon.h for
 more details.

 accessory:  If true, an instance of the application will be placed
 in the Accessories Palette.  The instance will have default parameters.
 You can also create customized accessories instances using the ACESSRY
 subdirectory.  See appmon.h for more details.

 hotMode:  If true, instances of the application are created in hot
 mode by default.  Note that you can change a document's hot mode flag
 at msgInit time (or at any other time) using msgAppSetHotMode.  See
 app.h for more details.

 allowEmbedding:  If true, instances of the application allow child
 applications to be embedded within them.  This parameter cannot be
 modified on a per-document basis.

 confirmDelete:  If true, PenPoint will ask for user confirmation before
 deleting any instance of the application.  This parameter cannot be
 modified on a per-document basis.

 deinstallable:  If false, users will be prevented from deinstalling
 the application class.

 systemApp:  If true, users will not see the application on the list
 of choices for importing documents.

 lowMemoryApp:  If false, users will be prevented from activating
 instances of the application when the system is low on memory.

 fullEnvironment:  If true, instance 0 of the application will have a
 full environment, including a resource list and floating window list.
 If false, these two items are destroyed, saving memory.  In general,
 if your application does no processing in instance 0 (i.e., it simply
 calls AppMonitorMain()), you should set fullEnvironment to false to
 save unneeded memory.

*/

typedef struct APP_MGR_FLAGS {

	U16		stationery			: 1;	// Put in stationery notebook.
	U16		accessory			: 1;	// Put in accessory palette.
	U16		hotMode				: 1;	// Create docs in hot mode. 
	U16		allowEmbedding		: 1;	// Allow child embedded apps.
	U16		confirmDelete		: 1;	// Confirm document deletes.
	U16		deinstallable		: 1;	// App class deinstallable.
	U16		systemApp			: 1;	// Disable imports into this app.
	U16		lowMemoryApp		: 1;	// Allow activation under low memory.
	U16		fullEnvironment		: 1;   	// Initialize instance 0 environment.
	U16		reserved1			: 7;	// Reserved.	   
	U16		reserved2			: 16;	// Reserved.

} APP_MGR_FLAGS, *P_APP_MGR_FLAGS;

/****  AppMgr Metrics and NEW Structure  ****/
/*
 * Public instance data for an installed application class.
 * Also the new structure for creating a new installed application class.
 */
typedef struct APP_MGR_METRICS {
// All fields are passed back from msgAppMgrGetMetrics.
// For msgNew: in=specified, out=passed back, na=not applicable (don't care).

	OBJECT			dir;						   // na: App monitor dir.
	OBJECT			appMonitor;					   // na: App monitor object.
	OBJECT			resFile;					   // na: App res file.
	OBJECT			iconBitmap;					   // na: Icon bitmap.
	OBJECT			smallIconBitmap;			   // na: Small icon bitmap.
	OBJECT			appWinClass;				   // in: always clsAppWin.
	RECT32			defaultRect;				   // in: Default rectangle
                                                   //      (in points).
	char			name[nameBufLength]; 		   // na: Application name.
	char			version[nameBufLength];		   // na: Version.
 	char			company[nameBufLength];		   // in: Company name.		
	char			defaultDocName[nameBufLength]; // in/out: Default
                                                   //      document name.
	P_STRING		copyright;					   // in: Copyright notice.
	OS_PROG_HANDLE	programHandle;				   // out: Program handle.
	U32				reserved[4];				   // na: Reserved.
	APP_MGR_FLAGS	flags;                         // in: Described above.

} APP_MGR_METRICS, *P_APP_MGR_METRICS,
  APP_MGR_NEW_ONLY, *P_APP_MGR_NEW_ONLY;

/****************************************************************************
 msgNew	    takes P_APP_MGR_NEW, returns STATUS
	category: class message
	Install a new application class.

 The fields you commonly set are:
    pArgs->object.uid:            your application class's uid
    pArgs->cls.pMsg:              your application class's method table
    pArgs->cls.ancestor:          your application class's ancestor 
                                  (usually clsApp)
    pArgs->cls.size:              size of a document's instance data
    pArgs->cls.newArgsSize:       size of the _NEW struct for the app class
    pArgs->appMgr.defaultRect:    rectangle to open doc to when floating
    pArgs->appMgr.company:        your company's name
    pArgs->appMgr.defaultDocName: name of new documents of this application
    pArgs->appMgr.copyright:      copyright notice
    pArgs->appMgr.flags:          (see description of flags above)

 clsAppMgr objects cannot be locked, so clsAppMgr forces 
 pArgs->object.key to 0.
*/

#define	appMgrNewFields			\
	classNewFields				\
	APP_MGR_NEW_ONLY			appMgr;

typedef struct APP_MGR_NEW {
	appMgrNewFields
} APP_MGR_NEW, *P_APP_MGR_NEW;

/****************************************************************************
 msgNewDefaults	    takes P_APP_MGR_NEW, returns STATUS
	category: class message
	Initializes APP_MGR_NEW structure to default values.

 Zeroes out pArgs->appMgr and sets
//{
	pArgs->object.cap |= objCapCall | objCapSend | objCapScavenge;

	pArgs->appMgr.flags.stationery	   = true;
	pArgs->appMgr.flags.accessory	   = false;
	pArgs->appMgr.flags.allowEmbedding = true;
	pArgs->appMgr.flags.confirmDelete  = true;
	pArgs->appMgr.flags.deinstallable  = true;
	pArgs->appMgr.flags.systemApp	   = false;
	pArgs->appMgr.flags.hotMode		   = false;
    pArgs->appMgr.appWinClass          = clsAppWin;

	// Default rect: 300 x 300 points, centered in theRootWindow
    WIN_METRICS wm;
	ObjCallRet(msgWinGetMetrics, theRootWindow, &wm, s);
	pArgs->appMgr.defaultRect.size.w   = 300;
	pArgs->appMgr.defaultRect.size.h   = 300;
	pArgs->appMgr.defaultRect.origin.x = (wm.bounds.size.w/2) -
						 			     (pArgs->appMgr.defaultRect.size.w/2);
	pArgs->appMgr.defaultRect.origin.y = (wm.bounds.size.h/2) -
									     (pArgs->appMgr.defaultRect.size.h/2);
//}
*/

/****************************************************************************
 msgAppMgrGetMetrics		takes P_APP_MGR_METRICS, returns STATUS
	category: class message
	Passes back the AppMgr metrics.
*/
#define msgAppMgrGetMetrics					MakeMsg(clsAppMgr, 1)

/****************************************************************************
 msgAppMgrCreate		takes P_APP_MGR_CREATE, returns STATUS
	Creates a directory entry for a new document.
	
 This message transitions a document from the Non-Existent state to the
 Created state.

 clsAppMgr creates a new file system directory entry for the new document,
 using the name im pName.  clsAppMgr also stamps the new directory with the
 application's class.

 If pName is pNull, clsAppMgr creates a unique name, based on the 
 application name.  If pName is not pNull, it points to a 
 client-allocated buffer that must be nameBufLength bytes long.

 After msgAppMgrCreate, the document will appear in the appropriate 
 table of contents or icon window. But the application instance itself
 will not be created until msgAppMgrActivate, which transitions the document
 from the Created state to the Activated state.

 Return Value
    stsFSNodeNotFound:  Invalid pArgs->locator
*/
#define msgAppMgrCreate						MakeMsg(clsAppMgr, 2)

typedef struct APP_MGR_CREATE {

	FS_LOCATOR	locator;		// Parent doc. uid must be of clsAppDir.
	P_STRING	pName;			// in/out: Name of new app.
	U32	 		sequence;		// Sequence of new app in parent app.			
	BOOLEAN		renumber;		// true=update global sequence #s.
	U32			reserved[2];	// reserved.

} APP_MGR_CREATE, *P_APP_MGR_CREATE;

/****************************************************************************
 msgAppMgrActivate	takes P_APP_MGR_ACTIVATE, returns STATUS
	Activates a document.

 This message transitions a document from the Created or Dormant state
 to the Activated state.

 clsAppMgr creates a new process for the document, and a new instance of
 the application class in the new process.  The Application Framework
 will then send the new application instance msgAppInit if the document
 was in the Created state, or msgAppRestore if the document was in the
 Dormant state.

 Return Value
    stsAppMgrLowMemNoActivate:  Document could not be activated due to
                                low memory conditions.
    stsFSNodeNotFound:          Invalid pArgs->locator.
*/
#define msgAppMgrActivate					MakeMsg(clsAppMgr, 3)

typedef struct APP_MGR_ACTIVATE {

	OBJECT		winDev;		// Window device to activate app on.
	FS_LOCATOR	locator;	// Location of doc to activate.
	OBJECT		parent;		// Parent doc uid.
	OBJECT		uid;		// out: activated doc uid.

} APP_MGR_ACTIVATE, *P_APP_MGR_ACTIVATE;

#define stsAppMgrLowMemNoActivate		MakeStatus(clsAppMgr, 3)

/****************************************************************************
 msgAppMgrMove	takes P_APP_MGR_MOVE_COPY, returns STATUS
	Moves a document to a new location.

 clsAppMgr will display the appropriate UI to show the progress of any
 time-consuming moves.

 If the move fails due to low memory, user cancellation, etc.,
 msgAppMgrMove will nevertheless return a value >= stsOK.  
 The user will have been notified of the condition via standard error
 messaging facilities.
*/
#define msgAppMgrMove						MakeMsg(clsAppMgr, 4)

typedef struct APP_MGR_MOVE_COPY_STYLE {

	U16	showConfirm         : 1;    // show confirmation UI
    U16 showProgress        : 1;    // show progress UI
    U16 reserved            : 14;	// reserved.
    U16 reserved2           : 16;	// reserved.

} APP_MGR_MOVE_COPY_STYLE, *P_APP_MGR_MOVE_COPY_STYLE;

typedef struct APP_MGR_MOVE_COPY {

	FS_LOCATOR				locator;			// Source document location.
	OBJECT					source;				// Source object.
	OBJECT					dest;				// Destination object.  
	XY32					xy;					// x,y location in dest object.
	CHAR					name[nameBufLength];// in:out New doc name;
	BOOLEAN					renumber;			// true=update global sequence #s.
	APP_MGR_MOVE_COPY_STYLE	style;				// Move/copy style.
	OBJECT					appWin;				// out: move/copied appwin.

} APP_MGR_MOVE_COPY, *P_APP_MGR_MOVE_COPY;

/****************************************************************************
 msgAppMgrCopy		takes P_APP_MGR_MOVE_COPY, returns STATUS
	Copies a document to a new location.  

 clsAppMgr will display the appropriate UI to show the progress of any
 time-consuming copies.

 If the copy fails due to low memory, user cancellation, etc.,
 msgAppMgrCopy will nevertheless return a value >= stsOK.  
 The user will have been notified of the condition via standard error
 messaging facilities.
*/
#define msgAppMgrCopy						MakeMsg(clsAppMgr, 5)

/****************************************************************************
 msgAppMgrFSMove	takes P_APP_MGR_FS_MOVE_COPY, returns STATUS
    category: internal use only
	Low-level move message used internally by msgAppMgrMove.
*/
#define msgAppMgrFSMove						MakeMsg(clsAppMgr, 17)

typedef struct APP_MGR_FS_MOVE_COPY {

	FS_LOCATOR		source;				// Source doc location.
	FS_LOCATOR		dest;				// Location of new parent doc.
	U32				sequence;			// Sequence of new doc in parent doc.
	CHAR			name[nameBufLength];// in/out: Name of new doc.
	U32				reserved[2];		// reserved.

} APP_MGR_FS_MOVE_COPY, *P_APP_MGR_FS_MOVE_COPY;

/****************************************************************************
 msgAppMgrFSCopy		takes P_APP_MGR_FS_MOVE_COPY, returns STATUS
    category: internal use only
	Low-level copy message used internally by msgAppMgrCopy.
*/
#define msgAppMgrFSCopy						MakeMsg(clsAppMgr, 18)


/****************************************************************************
 msgAppMgrDelete		takes P_APP_MGR_DELETE, returns STATUS
	Deletes a document.

 This message transitions a document from the Created or Dormant state
 to the Non-Existent state.  The document is deleted along with all of
 its directory nodes, embedded documents, document processes, and so on.
*/
#define msgAppMgrDelete						MakeMsg(clsAppMgr, 6)

typedef struct APP_MGR_DELETE {

	FS_LOCATOR	locator;		// Document to delete.
	BOOLEAN		renumber;		// true=update global sequence #s.
	U32			reserved[2];	// reserved.

} APP_MGR_DELETE, *P_APP_MGR_DELETE;

/****************************************************************************
 msgAppMgrRename		takes P_APP_MGR_RENAME, returns STATUS
	Renames a document.

 pName must point to a buffer nameBufLength long.

 Return Value
    stsAppBadName:          Invalid new name.
    stsAppDuplicateName:    Name already in use.
*/
#define msgAppMgrRename						MakeMsg(clsAppMgr, 7)

typedef struct APP_MGR_RENAME {

	FS_LOCATOR	locator;		// Location of app to rename.
	P_STRING	pName;			// in/out: New app name.
	U32			reserved[2];	// reserved.

} APP_MGR_RENAME, *P_APP_MGR_RENAME;

/****************************************************************************
 msgAppMgrShutdown 	takes P_FS_LOCATOR, returns STATUS
	Unconditionally shuts down an application instance and all children.

 This message transitions a document from the Activated or Opened state
 to the Dormant state.  The document is not given the opportunity to
 veto the shutdown.  The document is sent msgAppSave before the shutdown,
 so it can file its data.
*/
#define msgAppMgrShutdown					MakeMsg(clsAppMgr, 8)

/****************************************************************************
 msgAppMgrGetRoot	takes P_APP_MGR_GET_ROOT, returns STATUS
	Passes back the root application (clsRootContainerApp) of a tree of
    applications.
*/
#define msgAppMgrGetRoot					MakeMsg(clsAppMgr, 9)

typedef struct APP_MGR_GET_ROOT {

	FS_LOCATOR	locator;				// Location of app.
	char		path[fsPathBufLength];	// out: Path to root.	
	UUID		uuid;					// out: Root uuid;
	OBJECT		app;					// out: Root app.  objNull if inactive.
	U32			reserved[2];			// reserved.

} APP_MGR_GET_ROOT, *P_APP_MGR_GET_ROOT;

/****************************************************************************
 msgAppMgrSetIconBitmap	takes OBJECT, returns STATUS
	Specifies the large application icon bitmap.
*/
#define msgAppMgrSetIconBitmap				MakeMsg(clsAppMgr, 10)

/****************************************************************************
 msgAppMgrSetSmallIconBitmap	takes OBJECT, returns STATUS
	Specifies the small application icon bitmap.
*/
#define msgAppMgrSetSmallIconBitmap			MakeMsg(clsAppMgr, 11)

/****************************************************************************
 msgAppMgrRevert	takes P_FS_LOCATOR, returns STATUS
	Reverts a document to its most recently filed copy.
*/
#define msgAppMgrRevert						MakeMsg(clsAppMgr, 12)

/****************************************************************************
 msgAppMgrRenumber	takes P_FS_LOCATOR, returns STATUS
	Renumbers an application heirarchy.  
	
 The FS_LOCATOR must be a locator for a clsRootContainerApp.
*/
#define msgAppMgrRenumber					MakeMsg(clsAppMgr, 13)

/****************************************************************************
 msgAppMgrDumpSubtree	takes P_FS_LOCATOR, returns STATUS
	Dumps the attributes of a subtree of documents.

 The information is output to the debug window or device.
 The dumped fields for each node are:
    -:  document name
    -:  UUID (low 32 bits followed by high 32 bits)
    -:  old UUID (low 32 bits followed by high 32 bits)
    -:  application class
    -:  number of children
    -:  sequence number
*/
#define msgAppMgrDumpSubtree				MakeMsg(clsAppMgr, 14)

/****************************************************************************
 msgAppMgrGetResList	takes P_APP_MGR_GET_RES_LIST, returns STATUS
	Creates a resource list, given an application UUID.  
	
 The resource list will contain the document resource file, 
 the application resource file, the preference resource file, and 
 the system resource file.  resList should be set to objNull or 
 a well-known uid.
*/
#define msgAppMgrGetResList					MakeMsg(clsAppMgr, 15)

typedef struct APP_MGR_GET_RES_LIST {

	UUID		appUUID;	// App uuid.
	OBJECT		resList;	// in/out: resource file list.

} APP_MGR_GET_RES_LIST, *P_APP_MGR_GET_RES_LIST;


#endif	// APPMGR_INCLUDED

