/****************************************************************************
 File: app.h

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.44  $
   $Author:   pstahl  $
     $Date:   28 Feb 1992 15:32:12  $

 This file contains the API definition for clsApp.
 The functions described in this file are contained in APP.LIB. 

 clsApp inherits from clsObject. 
 Provides the standard behavior for a PenPoint application.

****************************************************************************/

/**** Introduction ****/
/*
 PenPoint applications rely on clsApp to create and display their main 
 window, save state, terminate the application instance, and so on.
 Every application developer needs to create a descendant of clsApp and 
 have the descendant handle a few important messages. See clsTemplateApp in
 \penpoint\sdk\sample\templtap for an example of those messages an 
 application typically must handle.

 When the user turns to a document in the notebook, the PenPoint Application 
 Framework creates an application instance to manage that document. 
 Throughout this header file and the rest of our documentation, we use
 the term "document" to refer to an instance of an application class. 

*/
#ifndef APP_INCLUDED
#define APP_INCLUDED

#ifndef FS_INCLUDED
#include <fs.h>
#endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef OBJECT APP, *P_APP;

#define AppDebug(v, e) DbgFlag('R', v, e)

/****  Well-known Filenames  ****/
/*
 The Application Framework looks for information and stores document data
 in a series of well-known filenames. One of these is:
	-:	appResFileName, the application's resource file for its icons, 
		quick help, user interface strings, and so on. 

 Each document in the Notebook has its own directory, containing
 a collection of files for the document's data and subdirectories for any
 embedded documents. These are:
	-:	appDocStateFileName, the resource file for any objects that the 
 		document saves. In general, this is called the document's resource
		file
	-:	appDocResFileName, a resource file for preferences, including 
		print metrics (once they are changed from the defaults) and 
		comments that the user wrote in the "Comments" option sheet 
	-:	appDocLinkFileName, the document's saved Reference Buttons and 
		descriptors for	what they are linked to
	-:	appActiveDocLinkFileName, a working document of newly created 
		(but not yet saved) Reference Buttons
	-:	appCorkboardDirName, the name of the subdirectory for documents
		embedded on the document's corkboard
	-:	subdirectories for any other embedded documents.
*/
#define appResFileName				"APP.RES"
#define appDocStateFileName			"DOCSTATE.RES"
#define appDocResFileName			"DOC.RES"
#define appDocLinkFileName			"DOC.LNK"
#define appActiveDocLinkFileName	"ACTDOC.LNK"
#define appCorkboardDirName			"CORKBD"

/****  Status Codes  ****/
/*
 These are the status codes returned by clsApp.
*/
#define stsAppRefused					MakeStatus(clsApp, 1)
#define stsAppMoveRCAppToCApp			MakeStatus(clsApp, 2)
#define stsAppMoveCAppToInvalid			MakeStatus(clsApp, 3)
#define stsAppCopyRCAppToCApp			MakeStatus(clsApp, 13)
#define stsAppCopyCAppToInvalid			MakeStatus(clsApp, 14)
#define stsAppNotMovable				MakeStatus(clsApp, 4)
#define stsAppNotCopyable				MakeStatus(clsApp, 5)
#define stsAppNotDeletable				MakeStatus(clsApp, 6)
#define stsAppDuplicateName				MakeStatus(clsApp, 7)
#define stsAppBadName					MakeStatus(clsApp, 17)
#define stsAppNotFound					MakeStatus(clsApp, 8)
#define stsAppOpened					MakeStatus(clsApp, 9)
#define stsAppNoSelection				MakeStatus(clsApp, 10)
#define stsAppSelRequestNotSupported	MakeStatus(clsApp, 11)
#define stsAppOutOfMemory				MakeStatus(clsApp, 15)
#define stsAppCrashed					MakeStatus(clsApp, 16)
#define stsAppOpenFailedSupressError	MakeStatus(clsApp, 18)
#define stsAppErrorStartingDoc			MakeStatus(clsApp, 19)
#define stsAppErrorEmbedPrintApply		MakeStatus(clsApp, 20)
#define stsAppErrorLeftPrintMargin		MakeStatus(clsApp, 21)
#define stsAppErrorRightPrintMargin		MakeStatus(clsApp, 22)
#define stsAppErrorTopPrintMargin		MakeStatus(clsApp, 23)
#define stsAppErrorBottomPrintMargin	MakeStatus(clsApp, 24)
#define stsAppErrorHeaderPrintMargin	MakeStatus(clsApp, 25)
#define stsAppErrorFooterPrintMargin	MakeStatus(clsApp, 26)

/****  Document States  ****/
/*
 A document can be in one of three states. When the user opens a document,
 its state becomes appOpened. Once the user closes it, the document's
 state can be either appTerminated or appActivated.

 There are conditions when, after the user closes a document, the document's 
 objects needs to stay around (and not be freed). Such conditions include
 when the document's access speed is set to accelerated (a.k.a., "hot mode")
 and when the document owns the selection. If a document is closed 
 but needs to stay active, its state is set to appActivated. 
 If there is no reason to keep a document around after it has been closed, 
 its state becomes appTerminated (and the document is freed soon thereafter).

 You can specify additional conditions for keeping a closed document active
 by handling msgAppTerminateOK. See the description of this message 
 for further details.
*/
#define appTerminated		0	// closed doc, on its way to being freed
#define appActivated		1	// closed doc, with a reason to stay active
#define appOpened			2	// opened doc

/****  App toggle  ****/
/*
 These are toggles used as parameters to various messages.
*/
#define appOff			0
#define appOn			1
#define appToggle		2

/****  Printing Flags  ****/
/*
 The Application Framework uses these flags when opening a document to
 print it and its embedded documents. The typical application developer
 does not need to use these flags. However, if you open your own 
 embedded documents, you should never pass on appPrintingTopLevel to them
 (even if you were opened with appPrintingTopLevel set).
*/
#define appPrinting			((U16)flag0)
#define	appPrintingTopLevel ((U16)flag1)

/****  App Flags  ****/
/*
 This structure defines the application flags.  They include
 the state of the document (see Document States above) and 
 other common booleans.  This structure is used in APP_METRICS
 and by APP_DIR_FLAGS (defined in appdir.h).
*/
typedef struct APP_FLAGS {

	U16		state			: 2;	// Document state.			
	U16		hotMode			: 1;	// True = app is in hot mode.
	U16		floating		: 1;	// True = app is floating.
	U16		printing		: 1;	// True = app is printing.
	U16		topLevel		: 1;	// True = app is printing as top level.
	U16		reserved1		: 10;	// Reserved.
	U16		reserved2		: 16;	// Reserved.

} APP_FLAGS, *P_APP_FLAGS;

/****  App Metrics  ****/
/*
 This structure defines the public instance data for clsApp.
 You get a copy of this structure when you send msgAppGetMetrics to an 
 application object. The fields of APP_METRICS are as follows:

 uuid:	The document's uuid. It is stamped as an attribute
 on the document directory (see appdir.h). You can pass a document's uuid 
 to clsDirHandle or clsAppDir in msgNew to create a handle to the 
 document directory.  

 dir:  	 An instance of clsAppDir. It is the handle to the filesystem 
 directory for a document.  

 parent:	An instance of clsApp. A document's parent is the document that
 activated it (see appmgr.h - msgAppMgrActivate). If the user opens a 
 document from the Notebook, the Notebook is the parent. If the opened
 document is embedded within another document, its parent is the embeddor.

 children:	An instance of clsObject. This represents a list of the 
 documents that this document activated. There is often a one-to-one
 correspondence between a document's children and its embedded documents.

 mainWin:	The document's main window. If this field is objNull when 
 a document receives msgAppInit, the document self sends msgAppProvideMainWin 
 to create one.

 floatingWins:	An instance of clsList. It is the list of 
 subordinate windows that are floating above a document (e.g., option 
 sheets). See msgAppAddFloatingWin and msgAppRemoveFloatingWin for 
 more info.  

 childAppParentWin:	The preferred parent window for embedded documents.

 resList:	An instance of clsResList.  It is list of clsResFile objects. 
 The default list consists of (1) a document resource file, 
 (2) an application resource file, (3) a preference resource file, and 
 (4) the system resource file. See resfile.h for more details.
 
 resFile:	The document's resource file (the same one as in the resList).

 flags:		Various flags for the document. See the discussion of APP_FLAGS
 given above.

*/
typedef struct APP_METRICS {

	UUID		uuid;					// App uuid. 
	OBJECT		dir;					// App directory.		
	OBJECT		parent;					// Parent app.			 
	OBJECT		children;				// Child apps observe this object.
	OBJECT		mainWin;				// App main window.		
	OBJECT		floatingWins;			// List of floating windows.
	OBJECT		childAppParentWin;		// Root of child app window tree.
	OBJECT		resList;				// Resource file list.
	OBJECT		resFile;				// Document resource file.
	U32			reserved[2];			// Reserved.
	APP_FLAGS	flags;					// Flags.

} APP_METRICS, *P_APP_METRICS;

/**** Enabling and Disabling SAMs ****/
/*
 In its handler for msgAppCreateMenuBar, clsApp creates several menus and 
 menu items that are part of PenPoint's standard user interface.  
 These menus and items are known collectively as PenPoint's Standard 
 Application Menus," or "SAMs" for short. The SAMs are identified by
 tags in apptag.h and are described in the PenPoint User Interface Design
 Reference.  
 
 In many cases, descendants of clsApp should be involved in deciding when
 the SAM menu items should be enabled or disabled.  Sometimes a
 descendant should completely remove an item from the SAM.

 To enable and disable the SAM items, clsApp handles msgControlProvideEnable
 (see control.h for a description this message). Specifically, clsApp:

	-:	Always enables:
//{
			tagAppMenuPrintSetup
			tagAppMenuAbout
			tagAppMenuCheckpoint
			tagAppMenuRevert
			tagAppMenuSearch
			tagAppMenuSpell
//}

	-:	Enables this if theUndoManager has transactions to undo (see undo.h):
//{
			tagAppMenuUndo
//}

	-:	Asks the appropriate mananger to enable or disable:
//{
			tagAppMenuPrint
			tagAppMenuSend
//}

	-:	Always disables:
//{
			tagAppMenuSelectAll
			any unrecognized tag
//}

 Here are some examples of how descendants might want to modify the SAMs or
 respond to msgControlProvideEnable:

    -:  Most applications should support all of the features in the
        SAMs.  (That's why they're part of PenPoint's standard UI.)  But
        for a variety of reasons, some applications won't support some
        standard PenPoint features.  These applications should remove the 
        menu item from the SAMs in their handler for msgAppCreateMenuBar.
        (See msgAppCreateMenuBar below.) Menu items that might not be 
		supported include:
//{
			tagAppMenuPrintSetup
			tagAppMenuSearch
			tagAppMenuSpell
			tagAppMenuUndo
			tagAppMenuPrint
			tagAppMenuSend
//}

    -:  Applications should handle msgControlProvideEnable and return
        false if there's no data in the application, true otherwise, for:
//{
			tagAppMenuSelectAll
//}

 Selection owners should respond to msgControlProvideEnable for
 tagAppMenuMove, tagAppMenuCopy and tagAppMenuDelete.  Here are some notes
 on the proper response.
	
    -:  If there is no data selected, then all three items should be disabled.  

    -:  If the application data is read-only, Move and Delete should be
        disabled.

    -:  In most other cases, the item should be enabled.
*/

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *			Messages													   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew	takes P_APP_NEW, returns STATUS
	category: class message
	Creates and initializes a new document.

 clsApp initializes the new document's instance data to default values.

 You should never send msgNew directly to clsApp or its descendants. 
 Sending msgNew is not sufficient to create a viable document. The 
 document must have its own process and directory, which msgNew does
 not create.  To create a viable document, send msgAppMgrCreate (or
 msgAppMgrCopy) followed by msgAppMgrActivate to the app's application
 manager. (Remember that the application manager's uid is the well-known
 uid for the application class.)

 Descendants: You should never handle msgNew directly. Instead, handle
 msgInit by initializing your instance data. The ancestor must be called
 before your msgInit handler.
*/
typedef struct APP_NEW_ONLY {

	FS_LOCATOR	locator;		// Document's location in the filesystem.	
	OBJECT		winDev;			// Window device.
	BOOLEAN		appMonitor;		// True if app monitor instance.
	U16			reserved1;		// Reserved.
	U32			reserved2[4];	// Reserved.

} APP_NEW_ONLY, *P_APP_NEW_ONLY;

#define	appNewFields	\
	objectNewFields		\
	APP_NEW_ONLY		app;

typedef struct APP_NEW {

	appNewFields

} APP_NEW, *P_APP_NEW;


/****************************************************************************
 msgNewDefaults			takes P_APP_NEW, returns STATUS
	category: class message
	Initializes an APP_NEW structure to default values.

 Zeroes out pArgs->app.

 Descendants: You should handle msgNewDefaults by initializing your _NEW 
 structure to default values. The ancestor must be called before your
 handler.
*/


/****************************************************************************
 msgFree 	takes nothing, returns STATUS
	Destroys a document.

 The document frees its instance data, its children, its main window,
 and any option sheets it has created. Its final step is to kill its
 process, which means that flow of control never returns from this
 message handler.

 Descendants: You should handle msgFree by destroying all objects and
 resources you have created. The ancestor must be called after your handler.
*/

/****************************************************************************
 msgFreeOK 	takes nothing, returns STATUS
	Checks to see if a document and its children are willing to be freed.

 This message is self sent as a result of msgDestroy being sent to the 
 document.
 
 A document can be freed if it can be terminated (see above description
 of Document States). To determine if it can be terminated, the document 
 self sends msgAppTerminateOK; if this message returns stsOK, the document
 then sends msgFreeOK to each active child document (those on the
 metrics.children list). If all of the children return stsOK, then the
 document can be terminated.

 Descendants: You normally do not handle this message. Instead, handle
 msgAppTerminateOK.

 Return Value
	stsOK:			If the document can be terminated.
	stsAppRefused:	If the document should not be terminated.
*/

/****************************************************************************
 msgAppActivate 	takes nothing, returns STATUS
	Activates a document and its children.

 This message prepares an application to receive such requests as becoming 
 available to the user (msgAppOpen) and searching for some data 
 (msgAppSearch).

 Descendants: You normally do not handle this message.
*/
#define msgAppActivate					MakeMsg(clsApp, 1)

/****************************************************************************
 msgAppInit		takes DIR_HANDLE, returns STATUS
	Creates a document's default data file and main window.

 This message is sent the first time a document is activated.  
 It performs one-time initializations.

 If the main window is objNull, the document creates the main window by 
 self sending msgAppProvideMainWin. If childAppParentWin is objNull, the
 document sets it to be the main window. The document also sets the main
 window title by self sending msgAppGetName, followed by msgAppSetName.

 Descendants: You should handle this message by performing one-time 
 initializations. This typically means creating any stateful objects 
 that will be filed. The ancestor should be called before your handler.
*/
#define msgAppInit						MakeMsg(clsApp, 2)

/****************************************************************************
 msgAppRestore 	takes nothing, returns STATUS
	Restores a document from its saved instance data.
 	
 The document opens its resource file (appDocStateFileName), reads its 
 instance data, and closes the file.  When it receives msgRestore, 
 the document reads its main window from the file.

 Descendants: You normally do not handle this message. Instead, you 
 should handle msgRestore (which is sent as a result of this message). 
*/
#define msgAppRestore					MakeMsg(clsApp, 3)

/****************************************************************************
 msgAppRestoreFrom		takes DIR_HANDLE, returns STATUS
	Restores a document from a specified directory.

 This message is just like msgAppRestore, except the document opens 
 the resource file (appDocStateFileName) located in DIR_HANDLE.

 Descendants: You normally do not handle this message. Instead, you
 should handle msgRestore (which is sent as a result of this message). 
*/
#define msgAppRestoreFrom				MakeMsg(clsApp, 4)

/****************************************************************************
 msgAppSave 	takes nothing, returns STATUS
	Saves a document to its working directory.

 The document self sends msgAppSaveChildren to save its children.
 Next, the document opens its resource file (appDocStateFileName), 
 writes its instance data, and closes the file. The document also saves 
 its link file. When it receives msgSave, the document writes its main 
 window to the file.
	
 Descendants: You normally do not handle this message.  Instead, you
 should handle msgSave to save your instance data.
*/
#define msgAppSave						MakeMsg(clsApp, 5)

/****************************************************************************
 msgAppSaveTo 	takes DIR_HANDLE, returns STATUS
	Saves a document to a specified directory.

 This message is just like msgAppSave, except the document opens 
 the resource file (appDocStateFileName) located in DIR_HANDLE.
	
 Descendants: You normally do not handle this message.  Instead, you
 should handle msgSave to save your instance data.
*/
#define msgAppSaveTo					MakeMsg(clsApp, 6)

/****************************************************************************
 msgAppSaveChildren 	takes nothing, returns STATUS
	Saves a document's children.

 The document self sends msgAppSaveChild to save each child document.
	
 Descendants: You normally do not handle this message.  
*/
#define msgAppSaveChildren				MakeMsg(clsApp, 7)

/****************************************************************************
 msgAppSaveChild 	takes APP, returns STATUS
	Saves the specified child document.

 The document sends msgAppSave to APP. 

 Descendants: You normally do not handle this message.  
*/
#define msgAppSaveChild					MakeMsg(clsApp, 97)

/****************************************************************************
 msgAppOpen 	takes P_APP_OPEN, returns STATUS
	Opens a document's main window. 

 If the document's main window has not been sized, the document sets it
 to the default size.  It also updates the 'parentWin' and 
 'childAppParentWin' fields in the application metrics. The document then
 sets its state to appOpened and self sends msgAppOpenChildren to
 open its child documents.

 This message is sent to the document when it is to be made 
 available to the user for direct interaction.

 Descendants: You should handle this message by creating any non-stateful
 objects that are necessary to display the document's UI. You should also 
 fill in 'childAppParentWin' - normally with the document's client window.

 You typically create the menu bar in response to this message.  Self send
 msgAppCreateMenuBar to create the menu bar, and then send msgFrameSetMetrics
 to your main window to insert the menu bar in the window.

 If you can't open the document, you should return stsFailed.  However, 
 if you have already displayed an error message to the user, then return
 stsAppOpenFailedSupressError.

 The ancestor should be called after your handler.
*/
#define msgAppOpen						MakeMsg(clsApp, 8)

typedef struct APP_OPEN {

	OBJECT		parentWin;			// Document's parent window.		
	OBJECT		childAppParentWin;	// out: Parent window for child apps.
	U16			printing;			// in:	See printing flags.

} APP_OPEN, *P_APP_OPEN;

/****************************************************************************
 msgAppClose	takes nothing, returns STATUS
	Closes a document's main window.

 The document extracts its main window from the window tree. It then
 sets the 'parentWin' field in the application metrics to objNull and
 sets its state to appActivated. To close its children, it
 self sends msgAppCloseChildren.

 Descendants: You should handle this message by destroying any objects that
 you created in msgAppOpen. If you created the menu bar in your msgAppOpen
 handler, then you should send msgFrameDestroyMenuBar to your main window.
 The ancestor should be called before your handler.

 This message is not an indication to terminate the document;
 it may be followed by other requests for services such as
 searching or re-opening.
*/
#define msgAppClose						MakeMsg(clsApp, 9)

/****************************************************************************
 msgAppSetMainWin	takes WIN, returns STATUS
	Specifies a document's main window.

 The document updates its metrics.mainWin field to point to pArgs. It does 
 not destroy the existing mainWin.

 Descendants: You normally do not handle this message.
*/
#define msgAppSetMainWin				MakeMsg(clsApp, 10)

/****************************************************************************
 msgAppSetChildAppParentWin 	takes WIN, returns STATUS
	Specifies the window that is used as the parent window for child
    documents.

 Descendants: You normally do not handle this message.
*/
#define msgAppSetChildAppParentWin		MakeMsg(clsApp, 11)

/****************************************************************************
 msgAppGetMetrics	takes P_APP_METRICS, returns STATUS
	Passes back a copy of the application metrics.

 Descendants: You normally do not handle this message.
*/
#define msgAppGetMetrics				MakeMsg(clsApp, 12)

/****************************************************************************
 msgAppDispatch 	takes nothing, returns STATUS
	Starts message dispatching.

 Descendants: You normally do not handle this message.
*/
#define msgAppDispatch 					MakeMsg(clsApp, 13)

/****************************************************************************
 msgAppRename 	takes P_STRING, returns STATUS
	Renames a document.  

 After msgAppRename is sent to the document, the Application Framework
 sends msgAppSetName to change the document's window title.

 Descendants: You normally do not handle this message. Instead, you might 
 want to handle msgAppSetName.
*/
#define msgAppRename 					MakeMsg(clsApp, 14)

/****************************************************************************
 msgAppSetName 	takes P_STRING, returns STATUS
	Specifies a document's displayed name (in its main window title).  

 This message does not actually rename the document; it only sets the 
 title of the document's main window. This message is sent to a document
 after it receives msgAppRename, which does rename the document.

 Descendants: You can handle this message by changing or adding to the
 string passed in.  The ancestor will take the new string and display it in
 the document's title.  The ancestor must be called after your handler.
*/
#define msgAppSetName 					MakeMsg(clsApp, 15)

/****************************************************************************
 msgAppGetName 	takes P_STRING, returns STATUS
	Passes back a document's name. 

 The document passes back its name (not its main window's title). Note that
 P_STRING must be nameBufLength long. 

 Descendants: You normally do not handle this message.
*/
#define msgAppGetName 					MakeMsg(clsApp, 16)

/****************************************************************************
 msgAppDelete		takes nothing, returns STATUS
	Deletes a document from the system.  

 The document deletes its appWin from its embeddor and sends
 msgAppMgrDelete to the document's class.  

 Descendants: You normally do not handle this message.

 Return Value
	stsAppRefused:	If metrics.flags.deletable is false.
*/
#define msgAppDelete					MakeMsg(clsApp, 17)

/****************************************************************************
 msgAppActivateChildren 	takes nothing, returns STATUS
	Activates a document's embedded documents.

 The document first activates the embedded documents that are stored in
 subdirectories of metrics.dir by self sending msgAppActivateChild for each
 child. It then self sends msgAppActivateCorkMarginChildren to activate the
 embedded documents that appear in the cork margin.

 Descendants: You normally do not handle this message.
*/
#define msgAppActivateChildren			MakeMsg(clsApp, 18)

/****************************************************************************
 msgAppActivateCorkMarginChildren 	takes nothing, returns STATUS
	Activates embedded documents that are in a document's cork margin.	

 The document self sends msgAppActivateChild for each embedded document 
 in the cork margin.

 Descendants: You normally do not handle this message.
*/
#define msgAppActivateCorkMarginChildren	MakeMsg(clsApp, 96)

/****************************************************************************
 msgAppActivateChild	takes P_APP_ACTIVATE_CHILD, returns STATUS
	Instantiates and activates an embedded document.

 This message sends msgAppMgrActivate to activate the specified 
 embedded document.

 Descendants: You normally do not handle this message.

 Return Value
	stsAppRefused:	If the child appDir.attrs.flags.disabled is true 
                    (see appdir.h).
*/
#define msgAppActivateChild				MakeMsg(clsApp, 19)

typedef struct APP_ACTIVATE_CHILD {

	P_STRING	pPath;			// Path of child relative to self.
	APP			uid;			// out: Child app uid.

} APP_ACTIVATE_CHILD, *P_APP_ACTIVATE_CHILD;

/****************************************************************************
 msgAppAddFloatingWin	takes WIN, returns STATUS
	Adds a window to a document's list of floating windows.

 Descendants: You normally do not handle this message.
*/
#define msgAppAddFloatingWin			MakeMsg(clsApp, 20)

/****************************************************************************
 msgAppRemoveFloatingWin	takes WIN, returns STATUS
	Removes a window from a document's list of floating windows.

 Descendants: You normally do not handle this message.
*/
#define msgAppRemoveFloatingWin			MakeMsg(clsApp, 21)

/****************************************************************************
 msgAppFindFloatingWin	takes P_APP_FIND_FLOATING_WIN, returns STATUS
	Finds the floating window on a document's list of floating windows
    that matches the specified tag.

 Descendants: You normally do not handle this message.

 Return Value
	stsOK:			If the floating window is found
 	stsNoMatch:		If the floating window cannot be found
*/
#define msgAppFindFloatingWin			MakeMsg(clsApp, 22)

typedef struct APP_FIND_FLOATING_WIN {

	TAG			tag;	// in: tag to find.
	OBJECT		win;	// out: matching window, or objNull if not found.

} APP_FIND_FLOATING_WIN, *P_APP_FIND_FLOATING_WIN;

/****************************************************************************
 msgAppGetRoot	takes P_APP, returns STATUS
	Passes back a document's root document (which is typically the Notebook).

 Descendants: You normally do not handle this message.
*/
#define msgAppGetRoot					MakeMsg(clsApp, 23)

/****************************************************************************
 msgAppSetParent	takes APP, returns STATUS
	Specifies a document's parent document.

 Descendants: You normally do not handle this message.
*/
#define msgAppSetParent					MakeMsg(clsApp, 24)

/****************************************************************************
 msgAppSetHotMode	takes BOOLEAN, returns STATUS
	Turns hot mode on or off for a document.

 Descendants: You normally do not handle this message.
*/
#define msgAppSetHotMode		 		MakeMsg(clsApp, 25)

/****************************************************************************
 msgAppSetReadOnly	takes BOOLEAN, returns STATUS
	Specifies a docuement's read only flag.

 Descendants: You normally do not handle this message.
*/
#define msgAppSetReadOnly				MakeMsg(clsApp, 26)

/****************************************************************************
 msgAppSetDeletable	takes BOOLEAN, returns STATUS
	Specifies a document's deletable flag.

 Descendants: You normally do not handle this message.
*/
#define msgAppSetDeletable				MakeMsg(clsApp, 27)

/****************************************************************************
 msgAppSetMovable	takes BOOLEAN, returns STATUS
	Specifies a document's movable flag. Not implemented.

 See Also
	msgAppDirSetFlags
*/
#define msgAppSetMovable				MakeMsg(clsApp, 28)

/****************************************************************************
 msgAppSetCopyable	takes BOOLEAN, returns STATUS
	Specifies a document's copyable flag. Not implemented.

 See Also
	msgAppDirSetFlags
*/
#define msgAppSetCopyable				MakeMsg(clsApp, 29)

/****************************************************************************
 msgAppTerminate	takes BOOLEAN, returns STATUS
	Terminates a document.

 If true is passed in, the document is given the chance to veto the
 termination. It does this by self sending msgFreeOK to see if it
 is okay to free the document. If it is okay, the document saves itself
 by self sending msgAppSave, and then frees itself by self sending 
 msgDestroy.

 If false is passed in, the document is not given the chance to veto.
 The document terminates itself and all of its children unconditionally.

 Descendants: You normally do not handle this message. This message is a 
 request, not a command, to terminate. It may be sent ANY number of times 
 while a document is active.  If you need to free objects when a document
 is terminated, you should handle msgFree. Furthermore, if you want to
 add conditions under which a document should not be terminated, handle
 msgAppTerminateOK.
*/
#define msgAppTerminate					MakeMsg(clsApp, 30)

/****************************************************************************
 msgAppOpenChildren		takes BOOLEAN, returns STATUS
	Opens all of the documents on a document's metrics.children list.
	
 If false is passed in, the document opens its child documents on screen
 by self sending msgAppOpenChild for each child.

 If true is passed in, it opens its child documents for printing as 
 embedded documents.

 Descendants: You normally do not handle this message.  
*/
#define msgAppOpenChildren				MakeMsg(clsApp, 31)

/****************************************************************************
 msgAppOpenChild		takes APP_OPEN_CHILD, returns STATUS
	Opens a specific child of a document.

 Opens the specified child document by creating a window for it and 
 then sending it msgAppOpen.

 Descendants: You normally do not handle this message.  
*/
#define msgAppOpenChild					MakeMsg(clsApp, 32)

typedef struct APP_OPEN_CHILD {

	OBJECT		app;				// Document to open.
	U16			printing;			// See printing flags.

} APP_OPEN_CHILD, *P_APP_OPEN_CHILD;

/****************************************************************************
 msgAppCloseChildren	takes nothing, returns STATUS
	Closes a document's children.

 The document self sends msgAppCloseChild for each of its child documents.

 Descendants: You normally do not handle this message.  
*/
#define msgAppCloseChildren				MakeMsg(clsApp, 89)

/****************************************************************************
 msgAppCloseChild		takes APP, returns STATUS
	Closes a specific child of a document.

 The document closes the specified child document by sending it msgAppClose.

 Descendants: You normally do not handle this message.  
*/
#define msgAppCloseChild				MakeMsg(clsApp, 90)

/****************************************************************************
 msgAppGetEmbeddor	takes P_APP, returns STATUS
	Passes back a document's direct parent in the file system heirarchy.  

 The document finds its direct parent in the filesystem and passes back
 a pointer to it in P_APP. If the parent is not active, P_APP is set to null.

 Descendants: You normally do not handle this message.  
*/
#define msgAppGetEmbeddor				MakeMsg(clsApp, 33)

/****************************************************************************
 msgAppTerminateOK	takes nothing, returns STATUS
	Checks if a document is willing to terminate.

 The document self sends this message as a result of msgAppTerminate(true).
 The document refuses if: (1) the document is opened, (2) the document is 
 in hot mode, or (3) the document or any object in the document owns the
 selection.  

 Descendants: You should handle this message if you have your own conditions
 under which to veto document termination. Typically you call the ancestor
 first. If the ancestor returns stsAppRefused, then you also return this
 value. However, if your ancestor returns stsOK, you check for your veto
 conditions and return either stsOK or stsAppRefused. 

 Return Value
	stsOK:			If the document can be terminated.
	stsAppRefused:	If the document should not be terminated.
*/
#define msgAppTerminateOK				MakeMsg(clsApp, 34)

/****************************************************************************
 msgAppGetEmbeddedWin		takes P_APP_GET_EMBEDDED_WIN, returns STATUS
	Finds the specified clsEmbeddedWin object within a document.

 The document recursively enumerates its children, searching for
 a clsEmbeddedWin object with a matching embeddedWinMetrics.uuid (see
 embedwin.h).

 Descendants: You should handle this message only if you are managing
 embedded windows that are not in the main window's window tree. Typically
 you call the ancestor first. If the ancestor passes back a non-null win,
 then you don't need to do anything. However, if the ancestor passes back
 objNull for the win, you should check for a clsEmbeddedWin with a matching
 uuid.
*/
#define msgAppGetEmbeddedWin			MakeMsg(clsApp, 35)

typedef struct APP_GET_EMBEDDED_WIN {

	UUID	uuid;		// in: embedded win's uuid.
	OBJECT	win;		// out: embedded win. Set to objNull if no match.

} APP_GET_EMBEDDED_WIN, *P_APP_GET_EMBEDDED_WIN;

/****************************************************************************
 msgAppGetAppWin		takes P_APP_GET_APP_WIN, returns STATUS
	Finds a clsAppWin object within a document.

 The document recursively enumerates its children, searching for
 a clsAppWin object with a matching appWinMetrics.appUUID (see appwin.h).

 Descendants: You should handle this message only if you are managing
 embedded windows that are not in the main window's window tree. Typically
 you call the ancestor first. If the ancestor passes back a non-null win,
 then you don't need to do anything. However, if the ancestor passes back
 objNull for the win, you should check for a clsAppWin with a matching uuid.
*/
#define msgAppGetAppWin					MakeMsg(clsApp, 36)

typedef struct APP_GET_APP_WIN {

	UUID	uuid;		// in: app win's uuid.
	OBJECT	win;		// out: app win. Set to objNull if no match.

} APP_GET_APP_WIN, *P_APP_GET_APP_WIN;

/****************************************************************************
 msgAppOwnsSelection		takes P_APP_OWNS_SELECTION, returns STATUS
	Tests if any object in a document owns the selection.	

 The document sets ownSelection to true if the selection 
 belongs to itself or one of its children (if checkChildren is true).

 Descendants: You normally do not handle this message.  
*/
#define msgAppOwnsSelection				MakeMsg(clsApp, 37)

typedef struct APP_OWNS_SELECTION {

	BOOLEAN		checkChildren;	// in: check child documents, too?
	BOOLEAN		ownSelection;	// out: true if doc(s) own the selection.

} APP_OWNS_SELECTION, *P_APP_OWNS_SELECTION;

/****************************************************************************
 msgAppOpenTo	takes U32, returns STATUS
	Opens a document to a specific state.

 If appOpenToNormal is passed in, the document sends msgAppOpenChild
 to its parent to open itself. If appOpenToFloating is passed in, the
 document self sends msgAppFloat to open itself.

 Descendants: You normally do not handle this message.  
*/
#define msgAppOpenTo					MakeMsg(clsApp, 38)

// States to pass to msgAppOpenTo
#define appOpenToNormal		0	// Open a doc in place.
#define appOpenToFloating	1	// Open a doc to floating.
#define	appOpenToNextState	2	// Goto next state.  Not Implemented.

/****************************************************************************
 msgAppCloseTo	takes U32, returns STATUS
	Closes a document to a specific state.

 Short description: you probably don't need to worry about this message.

 Long description: When the user taps on an embedded document icon, the
 document opens. If the user then double taps on the embedded document's
 title bar, the embedded document floats above its parent (allowing the
 user to resize it, without changing the layout of the parent). When the
 user closes the floating document, it "closes" to its next state (i.e., 
 open, but not floating). Closing it again closes the embedded document
 down to its icon.

 When the user closes an embedded document, the Application Framework 
 sends the document msgAppCloseTo, passing it appCloseToNextState. However,
 the Application Framework needs a mechanism to close an embedded document
 all the way down to its icon (e.g., when the user closes the parent
 document). In such cases, the Application Framework sends msgAppCloseTo
 to the document, passing appCloseToNormal.
 
 Descendants: You normally do not handle this message.  
*/
#define msgAppCloseTo					MakeMsg(clsApp, 39)

// States to pass to msgAppCloseTo
#define appCloseToNormal	0	// Close to icon.
#define appCloseToNextState	1 	// Close to next state.

/****************************************************************************
 msgAppHide		takes nothing, returns STATUS
	Hides an open document.

 This message is used to get a document and all its associated windows
 off the screen as quickly as possible. It is usually followed (via
 ObjectPost) by msgAppClose, which is a heavier-weight message.

 The document (1) sends msgWinExtract to all windows in 
 metrics.floatingWins, (2) sends msgWinExtract to metrics.mainWin, and
 (3) recursively sends msgAppHide to all documents on metrics.children.

 Descendants: You should handle this message if you have visible windows
 that are not children of the main window or in the floating window list.
 The ancestor should be called after your handler.
*/
#define msgAppHide		 				MakeMsg(clsApp, 40)

/****************************************************************************
 msgAppSetFloatingRect	takes P_RECT32, returns STATUS
	Specifies a document's floating size and position.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetFloatingRect			MakeMsg(clsApp, 41)

/****************************************************************************
 msgAppSetOpenRect	takes P_RECT32, returns STATUS
	Specifies a document's open size and position.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetOpenRect				MakeMsg(clsApp, 42)

/****************************************************************************
 msgAppGetOptionSheet		takes P_APP_GET_OPTION_SHEET, returns STATUS
	Passes back the requested option sheet of a document.

 If the requested option sheet has already been created, the document
 just passes back its uid. Otherwise, it creates the sheet by self sending 
 msgOptionCreateSheet. If the requested sheetTag is not tagAppDocOptSheet,
 the document self sends msgOptionAddCards to let descendants add
 option cards to the newly created sheet.

 Descendants: You normally do not handle this message. If you want to add
 other cards to the document's option sheets, you can handle msgAppAddCards.
*/
#define msgAppGetOptionSheet			MakeMsg(clsApp, 91)

typedef struct APP_GET_OPTION_SHEET {

	TAG		sheetTag;	// in: tag of option sheet.
	OBJECT	sheet;		// out: sheet uid.

} APP_GET_OPTION_SHEET, *P_APP_GET_OPTION_SHEET;

/****************************************************************************
 msgAppGetDocOptionSheetClient	takes P_OBJECT, returns STATUS
	Passes back the client for a document's option sheets.

 The document passes back its main window's client window.

 Descendants: You normally do not handle this message.  
*/
#define msgAppGetDocOptionSheetClient	MakeMsg(clsApp, 93)

/****************************************************************************
 msgAppAddCards		takes P_OPTION_TAG, returns STATUS
	Adds cards to the specified option sheet of a document.

 If the specified sheet is tagAppAboutOptSheet, the document adds the 
 "About Document" and "About Application" option cards to the sheet. 
 If the sheet is tagAppDocOptSheet, the document adds the "Controls,"
 "Access" and "Comments" cards. If the sheet is tagAppPrintSetupOptSheet,
 the document adds the "Print Layout" card.

 Descendants: You tend not to handle this message. However, you can handle
 it if you want to add cards to any of the document's option sheets.
*/
#define msgAppAddCards					MakeMsg(clsApp, 100)

/****************************************************************************
 msgAppShowOptionSheet	takes P_APP_SHOW_OPTION_SHEET, returns STATUS
	Shows or hides a document's option sheet.

 The Application Framework sends this message to show (or hide) any
 of a document's option sheets. It is sent when, for example, the user
 picks any of the option cards from the SAMs or draws the check gesture
 on a document's title, over a selection, or over an embedded document
 icon.

 If show is true, the document self sends msgAppGetOptionSheet to get 
 the requested option sheet. To display the sheet, the document sends
 msgOptionGetCards and msgOptionShowCardAndSheet to the sheet.

 If show is false, the document self sends msgAppFindFloatingWin and
 msgAppRemoveFloatingWin to find and then hide the requested option sheet.

 Descendants: You normally do not handle this message.  
*/
#define msgAppShowOptionSheet			MakeMsg(clsApp, 92)

typedef struct APP_SHOW_OPTION_SHEET {

	TAG			sheetTag;	// In: Option sheet tag.
	TAG			cardTag;	// In: Option card tag to initially show, or
							// null to show the top card.
	BOOLEAN		show;		// In: true = show, false = hide.
	OBJECT		sheet;		// Out: option sheet.

} APP_SHOW_OPTION_SHEET, *P_APP_SHOW_OPTION_SHEET;

/****************************************************************************
 msgAppApplyEmbeddeeProps	takes OBJECT, returns STATUS
	Applies Embedded Printing option card values to first level embeddees.

 Descendants: You normally do not handle this message.  
*/
#define msgAppApplyEmbeddeeProps	MakeMsg(clsApp, 98)

/****************************************************************************
 msgAppGetBorderMetrics		takes P_APP_BORDER_METRICS, returns STATUS
	Passes back a document's border metrics.

 Descendants: You normally do not handle this message.  
*/
#define msgAppGetBorderMetrics	 		MakeMsg(clsApp, 94)

// Border styles
#define appBorderNone		0
#define appBorderSingle		1
#define appBorderDouble		2
#define appBorderDashed		3

typedef struct APP_BORDER_METRICS {

	U16		controls		: 1;	// Out: true/false.
	U16		titleLine		: 1;	// Out: true/false.
	U16		menuLine		: 1;	// Out: true/false.
	U16		corkMargin		: 1;	// Out: true/false.
	U16		scrollMargins	: 1;	// Out: true/false.
	U16		borderStyle		: 4;	// Out: Border style.
	U16		reserved		: 7;

} APP_BORDER_METRICS, *P_APP_BORDER_METRICS;

/****************************************************************************
 msgAppSetControls	takes U32, returns STATUS
	Turns a document's controls on or off.

 If appOff is passed in, the document turns its controls off. 
 If appOn is passed in, the controls are turned on. If appToggle
 is passed in, the document will toggle the state of the controls.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetControls		 		MakeMsg(clsApp, 47)

/****************************************************************************
 msgAppSetPrintControls		takes BOOLEAN, returns STATUS
	Turns a document's screen decorations off for printing. 

 The document turns its controls off so that it can be printed. It
 leaves user-set borders on only if	the document is printing 
 itself as an embedded document (pArgs = false).

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetPrintControls			MakeMsg(clsApp, 99)

/****************************************************************************
 msgAppSetTitleLine		takes U32, returns STATUS
	Turns a document's title line on or off.

 If appOff is passed in, the document hides its title line.
 If appOn is passed in, the title line is displayed. If appToggle
 is passed in, the document toggles whether the title line is displayed.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetTitleLine		 		MakeMsg(clsApp, 44)

/****************************************************************************
 msgAppSetMenuLine	takes U32, returns STATUS
	Turns a document's menu bar on or off.

 If appOff is passed in, the document hides its menu bar.
 If appOn is passed in, the menu bar is displayed. If appToggle
 is passed in, the document toggles whether the menu bar is displayed.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetMenuLine		 		MakeMsg(clsApp, 45)

/****************************************************************************
 msgAppSetCorkMargin			takes U32, returns STATUS
	Turns a document's cork margin on or off.

 If appOff is passed in, the document hides its cork margin.
 If appOn is passed in, the cork margin is created (if it doesn't exist)
 and displayed. If appToggle is passed in, the document toggles whether
 the cork margin is displayed.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetCorkMargin				MakeMsg(clsApp, 48)

/****************************************************************************
 msgAppSetScrollBars	takes U32, returns STATUS
	Turns a document's scroll bars on or off.

 If appOff is passed in, the document hides its scroll bars.
 If appOn is passed in, the scroll bars are displayed. If appToggle
 is passed in, the document toggles whether the scroll bars are displayed.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetScrollBars	 			MakeMsg(clsApp, 46)

/****************************************************************************
 msgAppSetBorderStyle	takes U32, returns STATUS
	Specifies the border style.

 The possible values for pArgs are listed above in msgAppGetBorderMetrics.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetBorderStyle	 		MakeMsg(clsApp, 95)

/****************************************************************************
 msgAppRevert	takes BOOLEAN, returns STATUS
	Reverts to the filed copy of a document.

 The document reverts to its previously saved state. If true is passed in,
 the document displays a note, asking the user to confirm the action 
 first. If false is passed in, the document just does the action.

 Descendants: If you do not support revert, you should handle this
 message by returning stsAppRefused. On the other hand, if you support
 revert but you manage your own data files or use memory mapped files,
 then it may be necessary to handle this message by appropriately undoing
 all data modifications since the last save. The ancestor should be called
 before your handler.
*/
#define msgAppRevert					MakeMsg(clsApp, 49)

/****************************************************************************
 msgAppIsPageLevel	takes nothing, returns STATUS
	Asks a document if it shows up as a page in the Notebook (as opposed
	to being embedded).

 Descendants: You normally do not handle this message.  

 Return Value
	stsOK:			If the document is page-level (i.e., its embeddor 
					inherits from clsContainerApp or clsRootContainerApp).
	stsNoMatch:		If the document is not page-level.
*/
#define msgAppIsPageLevel				MakeMsg(clsApp, 50)

/****************************************************************************
 msgAppProvideMainWin	takes P_OBJECT, returns STATUS
	Asks a document to provide its main window.  
	
 This message is sent during msgAppInit. If pArgs points to objNull,
 the document creates a default frame of type clsFrame and 
 passes the frame's uid back in pArgs.

 Descendants: You should handle this message if you want to replace the
 default clsFrame main window. In such cases, you tend not to call 
 the ancestor.

 See Also
	msgAppCreateClientWin
*/
#define msgAppProvideMainWin			MakeMsg(clsApp, 51)

/****************************************************************************
 msgAppCreateLink	takes P_APP_LINK, returns STATUS
	Creates a link from a document to another document.  

 The uuid of the document to link to is passed in.  The document passes back
 a link handle, which is used by msgAppGetLink to retrieve the document.
 The document stores the uuid in its appDocLinkFileName file.

 Descendants: You normally do not handle this message.  
*/
#define msgAppCreateLink				MakeMsg(clsApp, 52)

typedef struct APP_LINK {

	UUID	appUUID;	// UUID of the document that is linked to.
	U32		link;		// Link handle.

} APP_LINK, *P_APP_LINK;

/****************************************************************************
 msgAppDeleteLink	takes P_APP_LINK, returns STATUS
	Deletes the specified link handle.

 Descendants: You normally do not handle this message.  
*/
#define msgAppDeleteLink				MakeMsg(clsApp, 53)

/****************************************************************************
 msgAppGetLink	takes P_APP_LINK, returns STATUS
	Passes back a document's UUID for the specified link handle.

 Descendants: You normally do not handle this message.  
*/
#define msgAppGetLink					MakeMsg(clsApp, 54)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Standard Application Menu Messages	  			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgAppCreateMenuBar	takes P_OBJECT, returns STATUS
	Creates the standard application menu bar.

 Descendants: You should handle this message by creating the document's
 menu bar.  If pArgs is non-null when the ancestor is called, clsApp
 will pre-pend the Document, Edit, and Option menus to the provided menu 
 bar. So you should call the ancestor after you make the menu bar. 
 After the ancestor returns, you can fix up the Document and Edit menus to 
 remove any buttons that you don't support or to add any new buttons.

 See the earlier description "Enabling and Disabling SAMs" for more details.
*/
#define msgAppCreateMenuBar				MakeMsg(clsApp, 55)

/****************************************************************************
 msgAppCreateClientWin	takes P_OBJECT, returns STATUS
	Creates a document's client window.

 The document creates a default client window of class clsEmbeddedWin and
 passes back its uid.

 The Application Framework does not send this message by default. 
 Instead, you should self send it at the appropriate time 
 (typically during msgAppInit, since the client window is usually stateful).

 Descendants: You should handle this message by creating your application-
 specific client window.  In such cases, you tend not to call your ancestor.
*/
#define msgAppCreateClientWin			MakeMsg(clsApp, 56)

/****************************************************************************
 msgAppSend		takes OBJECT, returns STATUS
	Sends a document.

 When the user taps on a button in the Send menu, the SAMs send this 
 message to the document, passing in theSendManager. The document 
 then self sends msgAppInvokeManager, passing on theSendManager.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSend						MakeMsg(clsApp, 57)

/****************************************************************************
 msgAppPrint		takes OBJECT, returns STATUS
	Prints a document.

 When the user issues the Print command (either by tapping on the Print
 button in the SAMs or by drawing the print gesture on the document's title
 line), the Application Framework sends this message to the document, 
 passing it thePrintManager. The document then self sends
 msgAppInvokeManager, passing on thePrintManager.

 Descendants: You normally do not handle this message.  
*/
#define msgAppPrint						MakeMsg(clsApp, 58)

/****************************************************************************
 msgAppPrintSetup		takes nothing, returns STATUS
	Displays a document's print setup option sheet.

 When the user taps on Print Setup, the SAMs send this message to the 
 document. The document self sends msgAppOptionShowOptionSheet, passing
 it tagAppPrintSetupOptSheet.

 Descendants: You normally do not handle this message.  
*/
#define msgAppPrintSetup				MakeMsg(clsApp, 59)

/****************************************************************************
 msgAppImport	takes nothing, returns STATUS
	Obsolete message. Not implemented.

 See Also
	msgImport  
*/
#define msgAppImport					MakeMsg(clsApp, 60)

/****************************************************************************
 msgAppExport	takes OBJECT, returns STATUS
	Prepares to export a document as a file.

 The document self sends msgAppInvokeManager, passing on pArgs.

 Descendants: You normally do not handle this message.  
*/
#define msgAppExport					MakeMsg(clsApp, 61)

/****************************************************************************
 msgAppAbout	takes nothing, returns STATUS
	Displays a document's "About" option sheet.

 When the user taps on About, the SAMs send this message to the document.
 The document self sends msgAppOptionShowSheet, passing it
 tagAppAboutOptSheet.

 Descendants: You normally do not handle this message. Instead, you should
 handle msgOptionAddCards by adding more cards to the About option sheet.
 Likewise, you should handle msgOptionProvideCard by modifying or adding
 specific controls to the standard About cards.
*/
#define msgAppAbout						MakeMsg(clsApp, 62)

/****************************************************************************
 msgAppHelp		takes nothing, returns STATUS
	Shows help for the application.  Not implemented - Reserved.

 Descendants: You should not handle this message. Instead, you can provide 
 help via resource files (see the Tic-Tac-Toe sample application for 
 an example).
*/
#define msgAppHelp						MakeMsg(clsApp, 63)

/****************************************************************************
 msgAppUndo		takes nothing, returns STATUS
	Undoes the previous operation on a document.

 The document sends msgUndoCurrent to theUndoManager. 

 Descendants: You normally do not handle this message.  Instead, see UNDO.H
 for information on how to undo your application's commands.
*/
#define msgAppUndo						MakeMsg(clsApp, 64)

/****************************************************************************
 msgAppMoveSel	takes nothing, returns STATUS
	Prepares to move a document's selection.

 When the user issues the Move command (either by tapping on Move in the 
 SAMs or by press-holding on a selection in the document), the Application
 Framework sends this message to the document. The document finds its
 selected object (by sending msgSelOwner to theSelectionManager) and then
 sends it msgSelBeginMove.

 Descendants: You normally do not handle this message.  
*/
#define msgAppMoveSel					MakeMsg(clsApp, 65)

/****************************************************************************
 msgAppCopySel	takes nothing, returns STATUS
	Prepares to copy the document's selection.

 When the user issues the Copy command (either by tapping on Copy in the 
 SAMs or by tap-press-holding on a selection in the document), the
 Application Framework sends this message to the document. The document 
 finds its selected object (by sending msgSelOwner to theSelectionManager)
 and then sends it msgSelBeginCopy.

 Descendants: You normally do not handle this message.  
*/
#define msgAppCopySel					MakeMsg(clsApp, 66)

/****************************************************************************
 msgAppDeleteSel	takes nothing, returns STATUS
	Deletes a document's selection.

 When the user issues the Delete command (either by tapping on Delete in 
 the SAMs or by drawing the delete gesture, the Application Framework
 sends this message to the document. The document gets its selected
 object (by sending msgSelOwner to theSelectionManager) and then
 sends it msgSelDelete.

 Descendants: You normally do not handle this message.  
*/
#define msgAppDeleteSel					MakeMsg(clsApp, 67)

/****************************************************************************
 msgAppSelOptions	takes nothing, returns STATUS
	Prepares to display the options for a document's selection. Obsolete.

 Descendants: You should not handle this message.  
*/
#define msgAppSelOptions				MakeMsg(clsApp, 68)

/****************************************************************************
 msgAppSelectAll	takes nothing, returns STATUS
	category: descendant responsibility
    Selects all of the objects in a document.

 When the user taps on Select All in the Standard Application Menu,
 the document self sends this message. 

 clsApp does not do anything in its message handler for this message.

 Descendants: You should handle this message and select everything 
 in the document. You tend not to call the ancestor.
*/
#define msgAppSelectAll					MakeMsg(clsApp, 69)

/****************************************************************************
 msgAppSearch	takes OBJECT, returns STATUS
	Searches a document for a string.

 When the user issues the Find command (either by tapping on Find in 
 SAMs or by drawing the find gesture on the document's title line), 
 the Application Framework sends this message to the document, passing
 it theSeachManager. In response, the document self sends 
 msgAppInvokeManager, passing on theSearchManager.
 
 Descendants: You normally do not handle this message.  
*/
#define msgAppSearch					MakeMsg(clsApp, 70)

/****************************************************************************
 msgAppSpell	takes OBJECT, returns STATUS
	Prepares to check a document's spelling.

 When the user issues the Spell command (either by tapping on Spell in 
 SAMs or by drawing the spell gesture on the document's title line), 
 the Application Framework sends this message to the document, 
 passing it theSpellManager. In response, the document self sends 
 msgAppInvokeManager, passing on theSpellManager.
 
 Descendants: You normally do not handle this message.  
*/
#define msgAppSpell						MakeMsg(clsApp, 71)

/****************************************************************************
 msgAppInvokeManager	takes OBJECT, returns STATUS
	Routes a message to a manager.  

 To route a standard application menu message to the object that 
 provides the behavior, the document self sends msgAppInvokeManager.
 The argument to the message is the well-known UID of the manager that
 performs the operation.  When the document receives msgAppInvokeManager,
 it sends msgAppExecute to the manager object.

 Descendants: You normally do not handle this message.  
*/
#define msgAppInvokeManager				MakeMsg(clsApp, 72)

/****************************************************************************
 msgAppExecute	takes P_APP_EXECUTE, returns STATUS
	Sent to the manager to execute the manager's behavior on a document.

 The document sends msgAppExecute to a manager when it receives 
 msgAppInvoke manager.  The manager performs some operation on the
 document or documents specified in the pArgs, such as printing,
 searching, or spell checking.

 Descendants: You normally do not handle this message.  
*/
#define msgAppExecute					MakeMsg(clsApp, 73)

typedef struct APP_EXECUTE {

	OBJECT	app;		// Requesting document.
	OBJECT	sel;		// Selected object.
	U32		count;		// Number of uuids.
	UUID	uuid[1];	// UUIDs of documents to operate on.

} APP_EXECUTE, *P_APP_EXECUTE;

/****************************************************************************
 msgAppExecuteGesture	takes P_GWIN_GESTURE, returns STATUS
	Invokes the default gesture behavior for a document's title line.
	
 Descendants: You normally do not handle this message.  However, if you want
 to handle a title line gesture differently than the default, you should
 handle this message. You tend not to call the ancestor.
*/
#define msgAppExecuteGesture			MakeMsg(clsApp, 74)

/****************************************************************************
 msgAppSetSaveOnTerminate	takes BOOLEAN, returns STATUS
	Tells a document to save itself before terminating.
	
 If msgAppSetSaveOnTerminate has been sent before msgAppTerminate, 
 the document will be sent msgAppSave even if it refuses to terminate.
 Normally, if a document vetos msgAppTerminate, it is not sent msgAppSave.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSetSaveOnTerminate		MakeMsg(clsApp, 75)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                       Notification messages  	       				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgAppTerminateConditionChanged	takes nothing, returns STATUS
	Try to terminate a document; sent when a terminate condition changed.  

 In response to this message, the document self sends msgAppTerminate(true).

 This message is self sent when a terminate condition has changed. For 
 example, the document might have given up its selection and can now
 be terminated.

 Descendants: You normally do not handle this message. Instead, 
 see msgAppTerminateOK.
*/
#define msgAppTerminateConditionChanged	MakeMsg(clsApp, 76)

/****************************************************************************
 msgAppSelChanged	takes BOOLEAN, returns STATUS
	Sent to a document when something in it becomes selected or deselected.

 pArgs is true when the document (or one of its embedded documents) gains
 the selection.  pArgs is false when the selection leaves the document.

 The document self sends msgAppTerminateConditionChanged when it no longer
 has the selection.

 Descendants: You normally do not handle this message.  
*/
#define msgAppSelChanged				MakeMsg(clsApp, 77)

/****************************************************************************
 msgAppOpened	takes APP_OPENED, returns STATUS
    category: observer notification
	Sent to observers of a document when the document is opened.

 pArgs->child is the uid of the document that has been opened.
*/
#define msgAppOpened					MsgNoError(MakeMsg(clsApp, 78))

/****************************************************************************
 msgAppClosed	takes APP_CLOSED, returns STATUS
    category: observer notification
	Sent to observers of a document when the document is closed.

 pArgs->child is the uid of the document that has been closed.
*/
#define msgAppClosed					MsgNoError(MakeMsg(clsApp, 79))

/****************************************************************************
 msgAppChildChanged	takes P_APP_CHILD_CHANGED, returns STATUS
    category: observer notification
	Sent to observers of a document when a child document is opened or closed.

 This message is sent to observers of a document in response to 
 msgAppOpened and msgAppClosed.
*/
#define msgAppChildChanged				MsgNoError(MakeMsg(clsApp, 80))

typedef struct APP_CHILD_CHANGED {

	OBJECT		parent;			// Parent of doc that changed.
	OBJECT		child;			// Doc that changed.
	UUID		uuid;			// UUID of doc that changed.
	MESSAGE		change;			// msgAppOpened or msgAppClosed.
	U32			reserved[4];	// Reserved.
	
} APP_CHILD_CHANGED, *P_APP_CHILD_CHANGED,
  APP_OPENED, *P_APP_OPENED,
  APP_CLOSED, *P_APP_CLOSED;

/****************************************************************************
 msgAppFloated	takes P_APP_FLOATED, returns STATUS
    category: observer notification
	Sent to observers when a document is floated or un-floated.
*/
#define msgAppFloated					MsgNoError(MakeMsg(clsApp, 81))

typedef struct APP_FLOATED {

	OBJECT		app;		// Document that is floated or un-floated.
	BOOLEAN		floatUp;	// true=document is floated.

} APP_FLOATED, *P_APP_FLOATED;

/****************************************************************************
 msgAppCreated	takes P_APP_CREATED, returns STATUS
    category: observer notification
	Sent to observers of clsApp when a document is created.
*/
#define msgAppCreated					MsgNoError(MakeMsg(clsApp, 82))

typedef struct APP_CREATED {

	OBJECT	rootContainer;		// Root container uid.
	UUID	rootContainerUUID;	// Root container uuid.
	UUID	uuid;				// Created doc's uuid.
	U32		reserved[4];		// Reserved.
	
} APP_CREATED, *P_APP_CREATED;

/****************************************************************************
 msgAppDeleted	takes P_APP_DELETED, returns STATUS
    category: observer notification
	Sent to observers of clsApp when a document is deleted.
*/
#define msgAppDeleted					MsgNoError(MakeMsg(clsApp, 83))

typedef struct APP_DELETED {

	OBJECT	rootContainer;		// Root container uid.
	UUID	rootContainerUUID;	// Root container uuid.
	OBJECT	app;				// Deleted document. objNull if inactive.
	UUID	uuid;				// Deleted document's uuid.
	U32		reserved[4];		// Reserved.

} APP_DELETED, *P_APP_DELETED;

/****************************************************************************
 msgAppMoved	takes P_APP_MOVED_COPIED, returns STATUS
    category: observer notification
	Sent to observers of clsApp when a document is moved.

 When a document is moved, the Application Framework notifies the
 observers of clsApp that a document has moved either a) within a root 
 container, or b) out of one root container and into another. (It may help
 you to remember that root containers are typically notebooks.)

 To notify the observers, the Application Framework creates a list 
 containing the document that is being moved and each of its embedded 
 documents. If the document is being moved within the root container, then
 for each of the documents in the list, the Application Framework sends 
 msgAppMoved to the observers of clsApp, specifying appMovedCopiedWithin.
 If the document is being moved from one container to another, the
 Application Framework sends msgAppMoved twice for each document, once
 specifying appMovedCopiedOutOf and once specifying msgMovedCopiedInto.

 See Also
    msgAppChanged 
*/
#define msgAppMoved						MsgNoError(MakeMsg(clsApp, 84))

// Move/copy values for moveCopyInfo argument
#define appMovedCopiedInto		0	// doc moved/copied to this root container
#define appMovedCopiedOutOf		1	// doc moved/copied from this root container
#define	appMovedCopiedWithin	2	// doc moved/copied within this root container

typedef struct APP_MOVED_COPIED {

	OBJECT		rootContainer;		// Root container uid.
	UUID		rootContainerUUID;	// Root container uuid.
	OBJECT		app;				// Moved/copied doc.  objNull if inactive.
	UUID		uuid;				// Moved/copied document's uuid.
	U32			moveCopyInfo;		// Type of move/copy.
	U32			reserved[4];		// Reserved.

} APP_MOVED_COPIED, *P_APP_MOVED_COPIED;

/****************************************************************************
 msgAppCopied	takes P_APP_MOVED_COPIED, returns STATUS
    category: observer notification
	Sent to observers of clsApp when a document is copied.

 When a document is copied, the Application Framework notifies the
 observers of clsApp that a document has been copied either a) within a root 
 container, or b) from one root container into another. (It may help
 you to remember that root containers are typically notebooks.)

 To notify the observers, the Application Framework creates a list 
 containing the document that is being copied and each of its embedded 
 documents. If the document is being copied within the root container, then
 for each of the documents in the list, the Application Framework sends
 msgAppCopied to the observers of clsApp, specifying appMovedCopiedWithin.
 If the document is being copied from one container to another, the
 Application Framework sends msgAppCopied twice for each document, once
 specifying appMovedCopiedOutOf and once specifying msgMovedCopiedInto.

 See Also
    msgAppChanged 
*/
#define msgAppCopied					MsgNoError(MakeMsg(clsApp, 85))

/****************************************************************************
 msgAppChanged	takes P_APP_CHANGED, returns STATUS
    category: observer notification
	Sent to observers of clsApp when a document has changed.

 This message is sent to observers of clsApp when a document has changed
 in some way (e.g., the document has moved, has a new name, has been
 created, and so on). 

 When a document is moved or copied, this message is sent to observers of
 clsApp. However, it is not sent for all of the document's embedded
 documents (thereby making it different from msgAppMoved and msgAppCopied). 

 See Also
    msgAppMoved
    msgAppCopied
*/
#define msgAppChanged					MsgNoError(MakeMsg(clsApp, 86))

// State of a doc's bookmark (which is interpreted in the NUI as a tab)
#define appBookmarkOn	1
#define appBookmarkOff	2

typedef struct APP_CHANGED {

	OBJECT	rootContainer;			// In: Root container uid.
	UUID	rootContainerUUID;		// In: Root container uuid.
	UUID	uuid;					// In: The uuid of the changed document.
	OBJECT	uid;					// In: objNull if changed doc was not active.
	U16		globalSequence	: 1;	// In: true if doc's container (i.e.,
									// notebook) needs to be renumbered.
	U16		name			: 1;	// In: true if doc's name changed
	U16		bookmark		: 2;	// In: new bookmark state, if changed
	U16		create			: 1;	// In: true if doc is new
	U16		deleted			: 1;	// In: true if doc was deleted
	U16		move			: 1;	// In: true if doc was moved
	U16		copy			: 1;	// In: true if doc was copied
	U16		reserved1		: 8;
	U16		moveCopyInfo;			// In: if doc was moved or copied, this
									// is set to move/copy value described
									// in msgAppMoved.
	U32		reserved2[4];

} APP_CHANGED, *P_APP_CHANGED;

/****************************************************************************
 msgAppInstalled 	takes CLASS, returns STATUS
    category: observer notification
	Sent to observers of clsApp when an application is installed.

 pArgs is the class of the application just installed.
*/
#define msgAppInstalled					MsgNoError(MakeMsg(clsApp, 87))

/****************************************************************************
 msgAppDeInstalled 	takes CLASS, returns STATUS
    category: observer notification
	Sent to observers of clsApp when an application is deinstalled.

 pArgs is the class of the application just deinstalled.
*/
#define msgAppDeInstalled				MsgNoError(MakeMsg(clsApp, 88))

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *						Public Functions								   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 AppMain	returns nothing
    Creates a document instance and starts dispatching messages to it.

 All developers should call AppMain from their main routine whenever
 processCount is greater than 0.
*/
STATUS EXPORTED AppMain			(void);

/****************************************************************************
 AppMonitorMain		returns nothing
    Creates an app monitor instance and handles installing the application.

 All developers should call AppMonitorMain from their main routine when 
 processCount is equal to 0. You specify the well-known uid of your
 application class and the well-known uid of your app monitor class. If
 you do not have an app monitor class, simply specify objNull for the
 second parameter.
*/
STATUS EXPORTED AppMonitorMain	(OBJECT, OBJECT);

#endif	// APP_INCLUDED


