/****************************************************************************
 $Workfile:   ANIMSP.H  $

 (C) Copyright 1992 by GO Corporation, All Rights Reserved.

 $Revision:   1.13  $
   $Author:   pstahl  $
     $Date:   20 Feb 1992 17:15:56  $

 This file contains the API definitions for clsAnimSPaper.

 clsAnimSPaper inherits from clsSPaper.
 Records pen strokes and plays them back at a reduced speed.  Provides
 settable speed, interstroke delay, line attribute and scaling parameters.

****************************************************************************/

/**** Introduction ****/
/*
 clsAnimSPaper "animates" the drawing of scribbles by painting a few
 points, then pausing for the specified number of milliseconds before
 continuing.  The animated playback is performed in a separate task, so
 playbacks will not disturb other events on the screen.  A semaphore is
 used internally to prevent multiple tasks from painting in the AnimSPaper
 window simultaneously.  The painting task is created whenever playback
 starts, and terminated when it finishes.

 The animation behavior is triggered by msgWinRepaint--that is, whenever
 the AnimSPaper is asked to paint itself.  This means that you'll get slow,
 "animated" painting regardless of the cause of the msgWinRepaint: layout,
 resize, scrolling, unclipping, and so forth.  If you want slow painting
 only under certain circumstances (e.g., when the user taps a button),
 set the Delay and Interstroke parameters to 0, then do this:

//{
    OS_MILLISECONDS om;

    om = yourDelay;
    ObjectCall(msgAnimSPaperSetDelay,       animSPaperInstance, &om);
    om = yourInterstroke;
    ObjectCall(msgAnimSPaperSetInterstroke, animSPaperInstance, &om);

    ObjectCall(msgWinDirtyRect,             animSPaperInstance, NULL);
    ObjectCall(msgWinUpdate,                animSPaperInstance, NULL);

    om = 0;
    ObjectCall(msgAnimSPaperSetDelay,       animSPaperInstance, &om);
    ObjectCall(msgAnimSPaperSetInterstroke, animSPaperInstance, &om);
//}

*/

/**** clsAnimSPaper Parameters ****/
/*
 There are four gettable/settable parameters having to do with scribble
 redisplay.  Delay specifies the number of milliseconds to wait between
 painting line segments.  It varies in inversely with the animation
 speed.  Interstroke delay is a separate delay to be used between scribble
 strokes.  It simulates the writer lifting and moving the pen from the
 end of one stroke to the beginning of the next.  Line sets the thickness
 and other attributes used in playing back the scribble.  Generally you
 shouldn't need to set anything except thickness.  Scale affects the size
 of the scribbles when they're played back.  The scale parameters will
 stretch/compress the scribble along the x and y axes, also scaling the
 scribble's distance from the lower-left corner (0,0). This is especially
 useful for applications which wish to scale in proportion to the system
 font size.  Note that since scribble scaling is in proportion to the
 original scribble, you may need to save what the system font size was
 when the scribble was recorded.

*/

/**** Other Facilities ****/
/*
 If pArgs->animSPaper.sendDone is true, an AnimSPaper will send
 msgAnimSPaperDone to its client when the animation is completed.

 For convenience two messages are provided to read and write scribbles
 to/from resource files.

*/

/**** Note on Delay and Interstroke Parameters ****/
/*
 AnimSPaper uses OSTaskDelay() to create the Delay and Interstroke delay.
 The minimum increment of OSTaskDelay is a system tick (systick), whose 
 length is device dependent.  Use OSSystemInfo() to find the length of a
 systick (see OS.H for details).  On an average PC or 386 system the
 systick is 55 milliseconds, or about an eighteenth of a second.  So
 micro-adjustments of Delay and Interstroke from, say, 60 milliseconds
 to 80 milliseconds will be ineffective.

*/

/**** Debugging Flags ****/
/*
 clsAnimSPaper uses the Handwriting debug flag set 'Z'.  clsAnimSPaper uses:

    80000:  Show all internal debugging messages

*/
#ifndef ANIMSP_INCLUDED
#define ANIMSP_INCLUDED

                                                    #ifndef SPAPER_INCLUDED
#include <spaper.h>     // ancestor flags
                                                    #endif
                                                    #ifndef SYSGRAF_INCLUDED
#include <sysgraf.h>    // line & scale def'ns
                                                    #endif
                                                    #ifndef FS_INCLUDED
#include <fs.h>         // filing def'ns
                                                    #endif

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *					Common #defines and typedefs						   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

typedef struct ANIM_SPAPER_NEW_ONLY {

	SYSDC_LINE	    line;	     // line attributes for scribble playback
	OS_MILLISECONDS delay;	     // delay between stroke segments on playback
                                 // (inverse of playback speed)
	OS_MILLISECONDS interstroke; // delay between strokes on playback
	OBJECT          client;      // recipient of msgAnimSPaperDone
    BOOLEAN         sendDone;    // if TRUE, animSPaper will send client
                                 // msgAnimSPaperDone when animation's done
	SCALE           scale;       // how much larger or smaller to scale the
                                 // scribble when it's played back. (1,1)
                                 // plays back at same scale as recorded.
	S32             spare1;      // unused (reserved)
	S32             spare2;      // unused (reserved)

} ANIM_SPAPER_NEW_ONLY,  *P_ANIM_SPAPER_NEW_ONLY;

#define	animSPaperNewFields		\
	sPaperNewFields			    \
	ANIM_SPAPER_NEW_ONLY	animSPaper;

typedef struct ANIM_SPAPER_NEW {
	animSPaperNewFields
} ANIM_SPAPER_NEW, *P_ANIM_SPAPER_NEW;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Messages									   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/****************************************************************************
 msgNew		takes P_ANIM_SPAPER_NEW, returns STATUS
	category: class message
	Creates an AnimSPaper window.

 The fields you commonly set are:
//{
	pArgs->animSPaper.line.thickness:   thickness of line on playback
	pArgs->animSPaper.delay:            inverse of animation speed 
	pArgs->animSPaper.interstroke:      delay between strokes
	pArgs->animSPaper.client:           whom to notify when animation is done
	pArgs->animSPaper.sendDone:         whether to notify client
	pArgs->animSPaper.scale:            playback size relative to input size
//}
*/

/****************************************************************************
 msgNewDefaults				takes P_ANIM_SPAPER_NEW, returns STATUS
	category: class message
	Initialize pArgs.

 Sets:
//{
	pArgs->animSPaper.line.cap        = sysDcCapRound;
	pArgs->animSPaper.line.join       = sysDcJoinRound;
	pArgs->animSPaper.line.thickness  = 6;
	pArgs->animSPaper.line.miterLimit = 10;
	pArgs->animSPaper.line.radius     = 0;
	pArgs->animSPaper.delay           = 40;
	pArgs->animSPaper.interstroke     = 160;
	pArgs->animSPaper.client          = objNull;
	pArgs->animSPaper.sendDone        = TRUE;
	pArgs->animSPaper.scale.x         = FxIntToFx(1);
	pArgs->animSPaper.scale.y         = FxIntToFx(1);
	pArgs->sPaper.flags              &= (~spScribbleEdit
                                       & ~spRedisplay
                                       & ~spVRuling
                                       & ~spRuling
                                       & ~spBackground);
	pArgs->win.flags.input           |= inputInkThrough;
//}
*/



/****************************************************************************
 msgAnimSPaperReadScribble		takes P_ANIM_SPAPER_SCRIBBLE, returns STATUS
	category: class message
	Reads a scribble from a resource file, sets it into the AnimSPaper
    and displays it.
*/
#define msgAnimSPaperReadScribble	MakeMsg(clsAnimSPaper, 1)

typedef struct ANIM_SPAPER_SCRIBBLE {

	FS_LOCATOR	locator;	// resource file locator
	RES_ID		resId;		// resource id for the scribble

} ANIM_SPAPER_SCRIBBLE, *P_ANIM_SPAPER_SCRIBBLE;

/****************************************************************************
 msgAnimSPaperWriteScribble		takes P_ANIM_SPAPER_SCRIBBLE, returns STATUS
	category: class message
	Writes the AnimSPaper's current scribble to a resource file.
*/
#define msgAnimSPaperWriteScribble	MakeMsg(clsAnimSPaper, 2)

/****************************************************************************
 msgAnimSPaperSetDelay		takes P_OS_MILLISECONDS, returns STATUS
	category: class message
    Specifies delay for scribble playback
*/
#define msgAnimSPaperSetDelay	MakeMsg(clsAnimSPaper, 4)

/****************************************************************************
 msgAnimSPaperGetDelay		takes P_OS_MILLISECONDS, returns STATUS
	category: class message
    Passes back delay for scribble playback
*/
#define msgAnimSPaperGetDelay	MakeMsg(clsAnimSPaper, 5)

/****************************************************************************
 msgAnimSPaperSetInterstroke		takes P_OS_MILLISECONDS, returns STATUS
	category: class message
    Specifies interstroke delay for scribble playback
*/
#define msgAnimSPaperSetInterstroke	MakeMsg(clsAnimSPaper, 6)

/****************************************************************************
 msgAnimSPaperGetInterstroke		takes P_OS_MILLISECONDS, returns STATUS
	category: class message
    Passes back interstroke delay for scribble playback
*/
#define msgAnimSPaperGetInterstroke	MakeMsg(clsAnimSPaper, 7)

/****************************************************************************
 msgAnimSPaperSetLine		takes P_SYSDC_LINE, returns STATUS
	category: class message
    Specifies line attributes for scribble playback
*/
#define msgAnimSPaperSetLine	MakeMsg(clsAnimSPaper, 8)

/****************************************************************************
 msgAnimSPaperGetLine		takes P_SYSDC_LINE, returns STATUS
	category: class message
    Passes back line attributes for scribble playback
*/
#define msgAnimSPaperGetLine	MakeMsg(clsAnimSPaper, 9)

/****************************************************************************
 msgAnimSPaperSetScale		takes P_SCALE, returns STATUS
	category: class message
    Specifies scaling for scribble playback.
    
 The scribble will be played back at a SCALE relative to the size at which
 it was recorded.  X and Y scales may be set independently.  The SCALE
 affects both the scribble and its distance from the lower-left corner (0,0).
*/
#define msgAnimSPaperSetScale	MakeMsg(clsAnimSPaper, 11)

/****************************************************************************
 msgAnimSPaperGetScale		takes P_SCALE, returns STATUS
	category: class message
    Passes back scaling for scribble playback
*/
#define msgAnimSPaperGetScale	MakeMsg(clsAnimSPaper, 12)

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Notifications   							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgAnimSPaperDone		takes OBJECT, returns STATUS
 	category:  advisory message
 	Sent to client when animation is complete.
    
 pArgs is the animSPaper's UID.  This message is sent only if there is a
 client and pArgs->animSPaper.sendDone was true at msgNew time.
*/
#define msgAnimSPaperDone		MakeMsg(clsAnimSPaper, 3)

//REFGEN BEGINIGNORE
/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *							Private Messages							   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/****************************************************************************
 msgAnimSPaperDCDone		takes OS_TASK_ID, returns STATUS
 	category:  internal use only
    Informs animSPaper that its drawing task has completed
*/
#define msgAnimSPaperDCDone		MakeMsg(clsAnimSPaper, 10)
//REFGEN ENDIGNORE

#endif
