/*  	
	File		QUEBIND.C
	Title		Queue management bindings for Lattice C
	Author(s)	Nick Webb
	Date		26.06.86

	Copyright (c) 1986 by Digital Research Inc.

Rev  	Date 	 Inits	Description
----------------------------------------------------------------------
1.0	26.06.86 NJW	Initial version
----------------------------------------------------------------------

*/

#include <portab.h>
#include "quebind.h"

#ifdef	I8086L
#define	LARGE_MODEL	1
#endif

extern	int	__DRIDOS(int, char *);		/* INT 224 bdos call */
extern	int	__extended_error;		/* Extended error */

#if LARGE_MODEL
extern	UWORD	__PTRSUB (QPB *, char *);	/* Convert s2 to offset from s1 */
extern	char	*__SYSDAT ();			/* Return sysdat pointer */
#endif

static	find_qpb();
static	is_valid(QPB *);
static	void	free_qpb(int);
static	char	*strncpy(char *, char *, int);


static	QDES	qd;				/* Queue descriptor */
static	QPB	q_array[NQUES];			/* Array of qpb's */

#if LARGE_MODEL
static	char	save_buffer[512];		/* L model buffer & RCT buffer */
static	int	q_length[16];			/* Array of queue lengths */
#endif

/*--------------------------------------------------------------*/
/*				q_make				*/
/*				======				*/
/* Action:							*/
/* The specified queue is opened.				*/
/* Returns:							*/
/* Error code		0	Successful			*/
/*			-1	No more queue descriptors	*/
/*--------------------------------------------------------------*/

q_make (name, size, depth)
register char	*name;				/* ASCIIZ string name of queue */
int	size;					/* Queue record size in bytes */
int	depth;					/* Depth in bytes */
{

	char	*s;				/* A.N.Other string pointer */

	if (find_qpb() == -1)			/* Find an unused qpb entry */
	    {
	    __extended_error = 7;		/* no more queue descriptors */
	    return (-1);
	    }
	qd.length = size;
	qd.nmsgs = depth;

	if ((*name == 'M') && (*(name + 1) == 'X'))
	    qd.flags = 1;
	else
	    qd.flags = 0;
	s = strncpy (qd.name, name, 8);		/* Max of 8 characters */
	while (s < (qd.name + 7))		/* IF less than 8 (<7) */
	    *s++ = ' ';
	if ((s == (qd.name + 7)) && (!*s))	/*	|| 7 */
	    *s = ' ';
	qd.buffer = 0;				/* Internal buffer space */
	return(__DRIDOS (Q_MAKE, (char *)&qd));
}


/*--------------------------------------------------------------*/
/*				q_open				*/
/*				======				*/
/* Action:							*/
/* Opens specified queue.					*/
/* Returns:							*/
/* qid		Queue handle					*/
/*--------------------------------------------------------------*/

q_open (name)
char	*name;					/* ASCIIZ string name of queue */
{

	int	qid;				/* Queue handle */
register QPB	*qptr;				/* QPB pointer */
register char	*s;				/* A.N.Other string pointer */

	if ((qid = find_qpb()) == -1)		/* Find the qpb entry */
	    {
	    __extended_error = 7;		/* no more queue descriptors */
	    return (-1);
	    }
	qptr = &q_array[qid];			/* Point to qpb */
	s = strncpy (qptr->name, name, 8);	/* Max of 8 characters */
	while (s < (qptr->name + 7))	 	/* if less than 8 (<7) */
	    *s++ = ' ';
	if ((s == (qptr->name + 7)) && !*s)	/*    || 7 */
	    *s = ' ';
	if (__DRIDOS (Q_OPEN, (char *)qptr))	/* IF a queue open error */
	    return (-1);			/*	Return error */
#if LARGE_MODEL
	if (qptr->resv1 == 0)			/* Get the queue length... */
	    {
	    s = (__SYSDAT() + qptr->queueid + 14);
	    q_length[qid] = *((int *) s);
	    }
	else
	    {					/* ELSE must be netted queue */
	    save_buffer[0] = 0xff;		/* RCT call to get net qcb */
	    if (__DRIDOS (N_RCT, save_buffer))	/* IF RCT call fails */
		{
		__extended_error = 9;		/*	Cannot find queue ! */
		free_qpb (qid);			/*	Free the queue */
		return (-1);			/*	And exit */
		}
	    s = save_buffer + 62 + ((qptr->queueid) - 1 * 20) + 18;
	    q_length[qid] = *((int *) s);
	    }
#endif
	return (qid);				/* Return queue handle */
}


/*--------------------------------------------------------------*/
/*				q_read				*/
/*				======				*/
/* Action:							*/
/* Read queue - this is an unconditional read, no return until	*/
/* data available.						*/
/* Returns:							*/
/* Error code		0	Successful			*/
/*		       -1	System error reading from queue	*/
/*--------------------------------------------------------------*/

q_read (qid, str)
int	qid;					/* Queue handle */
char	*str;					/* Character array to receive */
{						/*	one message */

register QPB	*qptr;				/* QPB pointer */

	qptr = &q_array[qid];			/* Point to the correct qpb */
	
	if (!is_valid (qptr))			/* IF NOT a valid qpb */
	    {
	    __extended_error = 9;		/*	Return 'Cannot find q' */
	    return (-1);
	    }
#if LARGE_MODEL
	qptr->buffer = __PTRSUB (q_array, save_buffer);	/* Set internal buffer */
	if (__DRIDOS (Q_READ, (char *)qptr) == -1)	/* IF error */
	    return (-1);
	strncpy (str, save_buffer, q_length[qid]);	/* Fill caller's buffer */
	return (0);
#else
	qptr->buffer = (UWORD) str;			/* Put in the buffer (offset) */
	return (__DRIDOS (Q_READ, (char *)qptr));	/* Do the system call */
#endif
}


/*--------------------------------------------------------------*/
/*				q_cread				*/
/*				=======				*/
/* Action:							*/
/* Conditional queue read.					*/
/* Returns:							*/
/* Error code		0	Successful			*/
/*			1	No message to read		*/
/*		       -1	System error reading from queue	*/
/*--------------------------------------------------------------*/

q_cread (qid, str)
int	qid;					/* Queue handle */
char	*str;					/* Character array to receive */
{						/*	one message */

register QPB	*qptr;				/* QPB pointer */

	qptr = &q_array[qid];			/* Point to the correct qpb */
	if (!is_valid (qptr))			/* IF NOT a valid qpb */
	    {
	    __extended_error = 9;		/*	Return 'Cannot find q' */
	    return (-1);
	    }

#if LARGE_MODEL
	qptr->buffer = __PTRSUB (q_array, save_buffer);	/* Set internal buffer */
#else
	qptr->buffer = (int) str;			/* Put in the buffer (offset) */
#endif

	if (__DRIDOS (Q_CREAD, (char *)qptr) == -1)	/* Do the system call */
	    if (__extended_error == 14)			/* If queue just empty */
		return (1);
	    else
		return (-1);				/* Otherwise a real error */
#if LARGE_MODEL
	strncpy (str, save_buffer, q_length[qid]);	/* Fill caller's buffer */
#endif
	return (0);					/* Ok */
}


/*--------------------------------------------------------------*/
/*				q_write				*/
/*				=======				*/
/* Action:							*/
/* Write to queue - This is an unconditional call, returns only */
/* after having written to queue.				*/
/* Returns:							*/
/* Error code		0	Successful			*/
/*		       -1	System error writing to queue	*/
/*--------------------------------------------------------------*/

q_write (qid, str)
int	qid;					/* Queue handle */
char	*str;					/* Character array to send */
{

register QPB	*qptr;				/* QPB pointer */

	qptr = &q_array[qid];			/* Point to the correct qpb */
	if (!is_valid (qptr))			/* IF NOT a valid qpb */
	    {
	    __extended_error = 9;		/*	Return 'Cannot find q' */
	    return (-1);
	    }

#if LARGE_MODEL
	qptr->buffer = __PTRSUB (q_array, save_buffer);	/* Set internal buffer */
	strncpy (save_buffer, str, q_length[qid]);	/* Fill save buffer */
#else
	qptr->buffer = (int) str;			/* Put in the buffer (offset) */
#endif

	return (__DRIDOS (Q_WRITE, (char *)qptr));	/* Do the system call */
}


/*--------------------------------------------------------------*/
/*				q_cwrite			*/
/*				========			*/
/* Action:							*/
/* Conditionally write to queue - write only if there is queue	*/
/* buffer space available.					*/
/* Returns:							*/
/* Error code		0	Successful			*/
/*			1	No queue buffer space		*/
/*		       -1	System error writing to queue	*/
/*--------------------------------------------------------------*/

q_cwrite (qid, str)
int	qid;					/* Queue handle */
char	*str;					/* Character array to send */
{

register QPB	*qptr;				/* QPB pointer */

	qptr = &q_array[qid];			/* Point to the correct qpb */
	if (!is_valid (qptr))			/* IF NOT a valid qpb */
	    {
	    __extended_error = 9;		/*	Return 'Cannot find q' */
	    return (-1);
	    }

#if LARGE_MODEL
	qptr->buffer = __PTRSUB (q_array, save_buffer);	/* Set internal buffer */
	strncpy (save_buffer, str, q_length[qid]);	/* Fill save buffer */
#else
	qptr->buffer = (int) str;			/* Put in the buffer (offset) */
#endif

	if (__DRIDOS (Q_CWRITE, (char *)qptr) == -1)	/* Do the system call */
	    if (__extended_error == 15)			/* If queue full */
		return (1);
	    else
		return (-1);				/* Otherwise a real error */
	return (0);					/* Ok */
}


/*--------------------------------------------------------------*/
/*				q_delete			*/
/*				========			*/
/* Action:							*/
/* Delete the queue from the system and reclaim queue buffer	*/
/* space. Any messages in the queue are lost.			*/
/* Returns:							*/
/* Error code		0	Successful			*/
/*		       -1	System error			*/
/*--------------------------------------------------------------*/

q_delete (qid)
int	qid;					/* Queue handle */
{

register QPB	*qptr;				/* QPB pointer */

	qptr = &q_array[qid];			/* Point to the correct qpb */
	if (!is_valid (qptr))			/* IF NOT a valid qpb */
	    {
	    __extended_error = 9;		/*	Return 'Cannot find q' */
	    return (-1);
	    }
	if (__DRIDOS (Q_DELETE, (char *)qptr))	/* IF any error */
	    return (-1);			/*	Return error */
	free_qpb (qid);				/* Free the qpb space */
	return (0);
}


/*--------------------------------------------------------------*/
/*				q_error				*/
/*				=======				*/
/* Action:							*/
/* Return the extended error value. This is only valid		*/
/* immediately following an error return.			*/
/* Returns:							*/
/* Error code							*/
/*--------------------------------------------------------------*/

q_error ()
{

	return (__extended_error);		/* Return last read error code */
}


/*--------------------------------------------------------------*/
/*				find_qpb			*/
/*				========			*/
/* Action:							*/
/* Find a free qpb buffer, return an index to it.		*/
/* Returns:							*/
/* QPB buffer number		Successful			*/
/* -1				Unsuccessful			*/
/*--------------------------------------------------------------*/

static	find_qpb ()
{

register int	i;				/* Index into QPB array */

	for (i = 0; i < NQUES; i++)
	    if (q_array[i].queueid == 0)
		return (i);
	return (-1);				/* No more queue buffers free */
}


/*--------------------------------------------------------------*/
/*				is_valid			*/
/*				========			*/
/* Action:							*/
/* Check qpb for being filled open / filled in.			*/
/* Returns:							*/
/* True or false						*/
/*--------------------------------------------------------------*/

static	is_valid (ptr)
QPB	*ptr;
{

	return ((ptr->queueid != 0));
}


/*--------------------------------------------------------------*/
/*				free_qpb			*/
/*				========			*/
/* Action:							*/
/* Free specified QPB for re-use.				*/
/* Returns:							*/
/* None.							*/
/*--------------------------------------------------------------*/

static	void	free_qpb(qid)
int	qid;
{

register QPB	*qptr;				/* Pointer to qpb array */
register int	i;				/* A.N.Other integer */

	qptr = &q_array[qid];
	qptr->resv1 = 0;			/* Make reserved words =0 */
	qptr->resv2 = 0;
	qptr->buffer = 0;			/* Zero buffer location */
	qptr->queueid = 0;			/* Queue id = 0 (unopened) */
	for (i = 0; i < 8; i++)			/* Blank out name */
	   qptr->name[i] = ' ';
}


/*--------------------------------------------------------------*/
/*				strncpy				*/
/*				=======				*/
/* Action:							*/
/* Copy string2 to string1 for max n characters. CAUTION this	*/
/* version of good ol strncpy will return a pointer to the END	*/
/* of the destination (or the null terminator if string < n).	*/
/* Returns:							*/
/* Pointer to the destination last character or NULL TERMINATOR.*/
/*--------------------------------------------------------------*/

static	char	*strncpy (s, t, n)
register char	*s, *t;				/* Source & destination strings */
int	n;					/* Max number to copy */
{

	while (*t && n--)			/* Repeat until end of string */
	    *s++ = *t++;			/*		or n = 0 */
	if (n)					/* IF string < n */
	    *s = '\0';				/* Null terminate the string */
	return (s);				/* Return pointer to the end */
}