/* 
 * Mach Operating System
 * Copyright (c) 1991,1990,1989 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Software Distribution Coordinator  or  Software.Distribution@CS.CMU.EDU
 *  School of Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

/* machterm.c -- mach terminal emulator
 *
 * Created by Zonnie L. Williamson at CMU, 1992
 *
 * This is a simple black and white terminal emulator.  The emulated
 * terminal is similar to a vt52 with the following features:
 *   - Multiple Font Sizes: 6x13, 7x13 and 9x13
 *   - Line Graphics
 *   - Bold, Underline and Standout Character Attributes
 *   - termcap of Termcap Description
 *   - Multiple Cursor Types
 *   - Black on White or White on Black Operation
 *
 * The terminal emulator operates on a bit-mapped display.  The display buffer
 * is one bit per pixel, 0=white and 1=black.  The display buffer is accessed
 * as an array of int's.  The following code demonstrates the how the pixels
 * are arranged in the buffer.  Note that position (x=0,y=0) is the upper left
 * corner of the display.
 *
 * void set_pixel(buffer, height, width, rowsize, x, y, value)
 * unsigned int *buffer;
 * int height, width, rowsize;
 * int x, y, value;
 * {
 *   if ((rowsize % sizeof(int)) != 0) return;
 *   if (width > (rowsize * 8)) return;
 *   if ((x < 0) || (x >= width)) return;
 *   if ((y < 0) || (y >= height)) return;
 *   bit = value << (sizeof(int) - 1 - x % sizeof(int));
 *   if (bit != 0)
 *     buffer[y * rowsize / sizeof(int) + x / sizeof(int)] |= bit;
 *   else
 *     buffer[y * rowsize / sizeof(int) + x / sizeof(int)] &= ~bit;
 * }
 *
 * The first two tests assert that the buffer can be accessed as an array of
 * int's and that the number of pixels in a row actually fits.
 * The second two tests assert that the specified point (x,y) is actually in
 * the display.
 * The value argument is 0 for a white pixel and 1 for a black pixel.  The
 * left shift moves the bit to the correct position within the int.
 * The final test chooses an OR operation to set the display bit and an AND
 * operation to clear the display bit.
 * The buffer index calculation first determines the index of the first int
 * of the row and then adds an offset to the correct int in the row.
 *
 * Character fonts are stored as an array of pixel rows, grouped by the font
 * height.  The array may be unsigned char or unsigned short depending on the
 * width of the font.  The character set consists of 128 characters.  The font
 * array contains the normal characters, followed by the bold characters.
 * For example, a 6x13 font is stored as unsigned char[(128 + 128) * 13] and
 * a 9x15 font is stored as unsigned short[(128 + 128) * 15].  The maximum
 * font width is 16.  Three fonts, 6x13, 7x13 and 9x15 are supported.  User
 * defined fonts are supported.
 *
 ******************************************************************************
 *
 * Three functions and a data type are exported.  They are:
 *   terminal_setup()    -- setup a display context
 *   terminal_buffer()   -- switch to a new display buffer
 *   terminal_emulator() -- display a character
 *   terminal_context_t  -- terminal context structure
 *
 * Note that the terminal_context_t is a simple character array.  The details
 * of the context structure are for internal use only.  One terminal context
 * structure is allocated internally.  It is used when a zero context pointer
 * is given.
 *   typedef char terminal_context_t[128];
 *
 * Use terminal_setup() to prepare a context for the terminal_emulator().
 * It returns non-zero if an error is detected. The arguments are:
 *   terminal_context     -- a pointer to a terminal context
 *   display_buffer       -- memory-mapped display, 1 bit per pixel
 *   display_height       -- display height, in pixels
 *   display_width        -- display width, in pixels
 *   display_rowsize      -- number of bytes in a display row
 *   font_bits            -- array of font bits
 *   font_height          -- font height, in pixels
 *   font_width           -- font width, in pixels
 *   initial_cursor       -- graphics character to use as a cursor
 *   white_on_black       -- if zero, black on white, otherwise white on black
 *   ungetc               -- write characters into the terminal input stream
 *   ungetc_argument      -- second argument to be given to ungetc()
 *   bell                 -- beep the bell
 *
 * extern int terminal_setup(terminal_context_t *terminal_context,
 *                           char *display_buffer,
 *                           int display_height,
 *                           int display_width,
 *                           int display_rowsize,
 *                           char *font_bits,
 *                           int font_height,
 *                           int font_width,
 *                           char initial_cursor,
 *                           int white_on_black,
 *                           void (*ungetc)(char, char*),
 *                           char *ungetc_argument,
 *                           void (*bell)(void));
 *
 * Use terminal_buffer() to switch to a new display buffer.  The contents of
 * the old buffer are copied into the new buffer.  It returns non-zero if an
 * error is detected.  The arguments are:
 *   terminal_context     -- a pointer to a terminal context
 *   display_buffer       -- memory-mapped display, 1 bit per pixel
 *   display_rowsize      -- number of bytes in a display row
 * extern int terminal_buffer(terminal_context_t *terminal_context,
 *                            char *display_buffer,
 *                            int display_rowsize);
 *
 * Use terminal_emulator() to display characters in a given terminal context.
 * The arguments are:
 *   terminal_context -- a pointer to a terminal context
 *   c                -- the character to display
 * extern void terminal_emulator(terminal_context_t *terminal_context, char c);
 *
 *
 ******************************************************************************
 *
 * This is the machterm character set.  Note that the codes 0-32 are only
 * accessable as the "alternate character set", offset by 95.
 * For example, the following sequence will display the Copyright symbol.
 *   machterm_putc(27);
 *   machterm_putc('n');
 *   machterm_putc(97);
 *   machterm_putc(27);
 *   machterm_putc('o');
 * The Escape-'n' selects the alternate character set.
 * The 97 is the Copyright symbol in the alternate character set.
 * Note that 97 = 95 + 2 = 'a'.
 * The Escape-'o' restores the normal character set.
 *
 * DEC  OCT  HEX
 * 000 0000 0x00 -undefined-
 * 001 0001 0x01 -undefined-
 * 002 0002 0x02 Copyright (graphics 'a')
 * 003 0003 0x03 Null Cursor (graphics 'b')
 * 004 0004 0x04 Closed Box Cursor (graphics 'c')
 * 005 0005 0x05 Open Box Cursor (graphics 'd')
 * 006 0006 0x06 Closed Oval Cursor (graphics 'e')
 * 007 0007 0x07 Open Oval Cursor (graphics 'f')
 * 008 0010 0x08 Closed Line Cursor (graphics 'g')
 * 009 0011 0x09 Open Line Cursor (graphics 'h')
 * 010 0012 0x0A -undefined-
 * 011 0013 0x0B Lower Right Corner (graphics 'j')
 * 012 0014 0x0C Upper Right Corner (graphics 'k')
 * 013 0015 0x0D Upper Left Corner (graphics 'l')
 * 014 0016 0x0E Lower Left Corner (graphics 'm')
 * 015 0017 0x0F Cross (graphics 'n')
 * 016 0020 0x10 High Line (graphics 'o')
 * 017 0021 0x11 Medium High Line (graphics 'p')
 * 018 0022 0x12 Horizontal Line (graphics 'q')
 * 019 0023 0x13 Medium Low Line (graphics 'r')
 * 020 0024 0x14 Low Line (graphics 's')
 * 021 0025 0x15 Tee Right (graphics 't')
 * 022 0026 0x16 Tee Left (graphics 'u')
 * 023 0027 0x17 Tee Up (graphics 'v')
 * 024 0030 0x18 Tee Down (graphics 'w')
 * 025 0031 0x19 Vertical Line (graphics 'x')
 * 026 0032 0x1A Less or Equal (graphics 'y')
 * 027 0033 0x1B Greater or Equal (graphics 'z')
 * 028 0034 0x1C Pi (graphics '{')
 * 029 0035 0x1D Not Equal (graphics '|')
 * 030 0036 0x1E -undefined- (graphics '}')
 * 031 0037 0x1F Degree (graphics '~')
 * 032 0040 0x20 Space
 * 033 0041 0x21 Exclamation Mark
 * 034 0042 0x22 Double Quote
 * 035 0043 0x23 Hash
 * 036 0044 0x24 Dollar
 * 037 0045 0x25 Percent
 * 038 0046 0x26 Ampersand
 * 039 0047 0x27 Single Quote
 * 040 0050 0x28 Left Paren
 * 041 0051 0x29 Right Paren
 * 042 0052 0x2A Star
 * 043 0053 0x2B Plus
 * 044 0054 0x2C Comma
 * 045 0055 0x2D Minus
 * 046 0056 0x2E Period
 * 047 0057 0x2F Slash
 * 048 0060 0x30 Zero
 * 049 0061 0x31 One
 * 050 0062 0x32 Two
 * 051 0063 0x33 Three
 * 052 0064 0x34 Four
 * 053 0065 0x35 Five
 * 054 0066 0x36 Six
 * 055 0067 0x37 Seven
 * 056 0070 0x38 Eight
 * 057 0071 0x39 Nine
 * 058 0072 0x3A Colon
 * 059 0073 0x3B Semicolon
 * 060 0074 0x3C Less Than
 * 061 0075 0x3D Equal
 * 062 0076 0x3E Greater Than
 * 063 0077 0x3F Question
 * 064 0100 0x40 At Sign
 * 065 0101 0x41 Uppercase A
 * 066 0102 0x42 Uppercase B
 * 067 0103 0x43 Uppercase C
 * 068 0104 0x44 Uppercase D
 * 069 0105 0x45 Uppercase E
 * 070 0106 0x46 Uppercase F
 * 071 0107 0x47 Uppercase G
 * 072 0110 0x48 Uppercase H
 * 073 0111 0x49 Uppercase I
 * 074 0112 0x4A Uppercase J
 * 075 0113 0x4B Uppercase K
 * 076 0114 0x4C Uppercase L
 * 077 0115 0x4D Uppercase M
 * 078 0116 0x4E Uppercase N
 * 079 0117 0x4F Uppercase O
 * 080 0120 0x50 Uppercase P
 * 081 0121 0x51 Uppercase Q
 * 082 0122 0x52 Uppercase R
 * 083 0123 0x53 Uppercase S
 * 084 0124 0x54 Uppercase T
 * 085 0125 0x55 Uppercase U
 * 086 0126 0x56 Uppercase V
 * 087 0127 0x57 Uppercase W
 * 088 0130 0x58 Uppercase X
 * 089 0131 0x59 Uppercase Y
 * 090 0132 0x5A Uppercase Z
 * 091 0133 0x5B Left Bracket
 * 092 0134 0x5C Back Slash
 * 093 0135 0x5D Right Bracket
 * 094 0136 0x5E Caret
 * 095 0137 0x5F Underscore
 * 096 0140 0x60 Back Quote
 * 097 0141 0x61 Lowercase A
 * 098 0142 0x62 Lowercase B
 * 099 0143 0x63 Lowercase C
 * 100 0144 0x64 Lowercase D
 * 101 0145 0x65 Lowercase E
 * 102 0146 0x66 Lowercase F
 * 103 0147 0x67 Lowercase G
 * 104 0150 0x68 Lowercase H
 * 105 0151 0x69 Lowercase I
 * 106 0152 0x6A Lowercase J
 * 107 0153 0x6B Lowercase K
 * 108 0154 0x6C Lowercase L
 * 109 0155 0x6D Lowercase M
 * 110 0156 0x6E Lowercase N
 * 111 0157 0x6F Lowercase O
 * 112 0160 0x70 Lowercase P
 * 113 0161 0x71 Lowercase Q
 * 114 0162 0x72 Lowercase R
 * 115 0163 0x73 Lowercase S
 * 116 0164 0x74 Lowercase T
 * 117 0165 0x75 Lowercase U
 * 118 0166 0x76 Lowercase V
 * 119 0167 0x77 Lowercase W
 * 120 0170 0x78 Lowercase X
 * 121 0171 0x79 Lowercase Y
 * 122 0172 0x7A Lowercase Z
 * 123 0173 0x7B Left Brace
 * 124 0174 0x7C Vertical Bar
 * 125 0175 0x7D Right Brace
 * 126 0176 0x7E Tilde
 * 127 0177 0x7F -undefined-
 */

/*****************************************************************************/

/* built-in 6x13 font */

/* This macro builds 6-bit wide rows. */
#define X(a, b, c, d, e, f) ((a<<5)|(b<<4)|(c<<3)|(d<<2)|(e<<1)|(f<<0))

static unsigned char font_6x13[256 * 13] = {
  /* 6x13 normal character 0 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 1 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 2 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,0,0,0,0,1),
  X(1,0,1,1,0,1),
  X(1,0,1,0,0,1),
  X(1,0,1,0,0,1),
  X(1,0,1,1,0,1),
  X(1,0,0,0,0,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 3 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 4 */
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  /* 6x13 normal character 5 */
  X(1,1,1,1,1,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,1,1,1,1,1),
  /* 6x13 normal character 6 */
  X(0,1,1,1,1,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  /* 6x13 normal character 7 */
  X(0,1,1,1,1,0),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(0,1,1,1,1,0),
  /* 6x13 normal character 8 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  /* 6x13 normal character 9 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,0,0,0,0,1),
  X(1,1,1,1,1,1),
  /* 6x13 normal character 10 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 11 */
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(1,1,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 12 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  /* 6x13 normal character 13 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,1),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  /* 6x13 normal character 14 */
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 15 */
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  /* 6x13 normal character 16 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 17 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 18 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 19 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 20 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 21 */
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,1,1,1),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  /* 6x13 normal character 22 */
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(1,1,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  /* 6x13 normal character 23 */
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 24 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  /* 6x13 normal character 25 */
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  /* 6x13 normal character 26 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,0,1,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 27 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 28 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(1,0,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 29 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,0),
  X(1,1,1,1,1,0),
  X(0,0,1,0,0,0),
  X(1,1,1,1,1,0),
  X(1,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 30 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,0,0),
  X(1,1,1,1,0,0),
  X(0,0,1,0,0,0),
  X(1,1,1,1,0,0),
  X(1,0,1,0,1,0),
  X(1,1,1,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 31 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 32 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 33 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 34 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 35 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(1,1,1,1,1,0),
  X(0,1,0,1,0,0),
  X(1,1,1,1,1,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 36 */
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,1,0,0,0),
  X(1,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,1,0,1,0),
  X(0,0,1,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 37 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,0,0,1,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(1,0,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 38 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,1,0,0,0),
  X(1,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,1,0,0,0),
  X(1,0,0,1,1,0),
  X(1,0,0,1,0,0),
  X(0,1,1,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 39 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 40 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 41 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 42 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(1,0,1,0,1,0),
  X(1,1,1,1,1,0),
  X(0,1,1,1,0,0),
  X(1,1,1,1,1,0),
  X(1,0,1,0,1,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 43 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 44 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 45 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 46 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 47 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 48 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 49 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,0,0,0),
  X(1,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 50 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 51 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 52 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(1,0,0,1,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 53 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,1,1,0,0),
  X(1,1,0,0,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 54 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 55 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 56 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 57 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 58 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 59 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 60 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 61 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 62 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 63 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 64 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,1,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,1,0,0),
  X(1,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 65 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,1,1,1,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 66 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(1,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 67 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 68 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,1,0),
  X(1,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 69 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 70 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 71 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,1,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 72 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,1,1,1,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 73 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 74 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(1,0,0,1,0,0),
  X(0,1,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 75 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,1,0,0),
  X(1,0,1,0,0,0),
  X(1,1,0,0,0,0),
  X(1,0,1,0,0,0),
  X(1,0,0,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 76 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 77 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,1,0,1,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 78 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,1,0,0,1,0),
  X(1,1,0,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,0,1,1,0),
  X(1,0,0,1,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 79 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 80 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 81 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,1,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 82 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,1,1,1,0,0),
  X(1,0,1,0,0,0),
  X(1,0,0,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 83 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 84 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 85 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 86 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 87 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,1,0,1,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 88 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 89 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 90 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 91 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 92 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 93 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 94 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,1,0,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 95 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 96 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 97 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,1,0),
  X(0,1,1,1,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 98 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 99 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 100 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(0,1,1,1,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 101 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,1,1,1,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 102 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,0,0,1,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(1,1,1,1,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 103 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  /* 6x13 normal character 104 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,1,1,0,0),
  X(1,1,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 105 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 106 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(0,0,0,1,0,0),
  X(1,0,0,1,0,0),
  X(1,0,0,1,0,0),
  X(0,1,1,0,0,0),
  /* 6x13 normal character 107 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,1,0,0),
  X(1,0,1,0,0,0),
  X(1,1,0,0,0,0),
  X(1,0,1,0,0,0),
  X(1,0,0,1,0,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 108 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 109 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 110 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,1,1,0,0),
  X(1,1,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 111 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 112 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,1,1,1,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  /* 6x13 normal character 113 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  X(0,0,0,0,1,0),
  /* 6x13 normal character 114 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,1,1,0,0),
  X(1,1,0,0,1,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 115 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(1,0,0,0,1,0),
  X(0,1,1,0,0,0),
  X(0,0,0,1,0,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 116 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(1,1,1,1,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,0,0),
  X(0,1,0,0,1,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 117 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,1,1,0),
  X(0,1,1,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 118 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,0,1,0,0),
  X(0,1,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 119 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 120 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(0,1,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,1,0,0),
  X(1,0,0,0,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 121 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(1,0,0,1,1,0),
  X(0,1,1,0,1,0),
  X(0,0,0,0,1,0),
  X(1,0,0,0,1,0),
  X(0,1,1,1,0,0),
  /* 6x13 normal character 122 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,1,0,0),
  X(0,0,1,0,0,0),
  X(0,1,0,0,0,0),
  X(1,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 123 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,1,1,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(1,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 124 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 125 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,0,1,1,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(0,0,1,0,0,0),
  X(1,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 126 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,0,0,1,0),
  X(1,0,1,0,1,0),
  X(1,0,0,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 normal character 127 */
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  /* 6x13 bold character 0 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 1 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 2 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,0,0,0,0,1),
  X(1,0,1,1,0,1),
  X(1,0,1,0,0,1),
  X(1,0,1,0,0,1),
  X(1,0,1,1,0,1),
  X(1,0,0,0,0,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 3 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 4 */
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  /* 6x13 bold character 5 */
  X(1,1,1,1,1,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,1,1,1,1,1),
  /* 6x13 bold character 6 */
  X(0,1,1,1,1,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  /* 6x13 bold character 7 */
  X(0,1,1,1,1,0),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(1,0,0,0,0,1),
  X(0,1,1,1,1,0),
  /* 6x13 bold character 8 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  /* 6x13 bold character 9 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,0,0,0,0,1),
  X(1,1,1,1,1,1),
  /* 6x13 bold character 10 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 11 */
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(1,1,1,1,0,0),
  X(1,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 12 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,0,0),
  X(1,1,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  /* 6x13 bold character 13 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,1),
  X(0,0,1,1,1,1),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  /* 6x13 bold character 14 */
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,1,1),
  X(0,0,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 15 */
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  /* 6x13 bold character 16 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 17 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 18 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 19 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 20 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 21 */
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,1,1),
  X(0,0,1,1,1,1),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  /* 6x13 bold character 22 */
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(1,1,1,1,0,0),
  X(1,1,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  /* 6x13 bold character 23 */
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 24 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  /* 6x13 bold character 25 */
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  /* 6x13 bold character 26 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 27 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 28 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(1,1,0,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 29 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,1),
  X(1,1,1,1,1,1),
  X(0,0,1,1,0,0),
  X(1,1,1,1,1,1),
  X(1,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 30 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,1,1,0,1,1),
  X(0,1,1,0,0,0),
  X(1,1,1,1,1,0),
  X(0,1,1,0,0,0),
  X(1,1,1,1,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 31 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 32 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 33 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 34 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 35 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 36 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(1,1,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,1,1),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 37 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,0,1,1),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,1,1,1,1),
  X(1,1,0,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 38 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,0,0,0),
  X(1,1,1,1,0,0),
  X(1,1,1,1,0,0),
  X(0,1,1,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,0,1,1,0),
  X(0,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 39 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 40 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 41 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 42 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 43 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(1,1,1,1,1,1),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 44 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 45 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 46 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 47 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 48 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 49 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,0,0),
  X(1,1,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 50 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 51 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 52 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,1,0,1,1,0),
  X(1,1,0,1,1,0),
  X(1,1,1,1,1,1),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 53 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,1,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 54 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,1,1,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,1,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 55 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 56 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 57 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,1,1,1),
  X(0,1,1,1,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 58 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 59 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 60 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(1,1,0,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 61 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 62 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(1,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 63 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 64 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,0),
  X(1,1,0,0,0,0),
  X(0,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 65 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 66 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(0,1,1,1,1,0),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 67 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 68 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(0,1,1,0,1,1),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 69 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 70 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 71 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,1,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 72 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 73 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 74 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 75 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,1,1,0),
  X(1,1,1,1,0,0),
  X(1,1,1,0,0,0),
  X(1,1,1,1,0,0),
  X(1,1,0,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 76 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 77 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 78 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,1,0,1,1),
  X(1,1,1,0,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,0,1,1,1),
  X(1,1,0,1,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 79 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 80 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 81 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 82 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,0),
  X(1,1,1,1,0,0),
  X(1,1,0,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 83 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 84 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 85 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 86 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 87 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 88 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 89 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 90 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 91 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 92 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 93 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 94 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 95 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 96 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 97 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,1,1),
  X(0,1,1,1,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,1,1,1),
  X(0,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 98 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,1,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,0,1,1),
  X(1,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 99 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 100 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,1,1,1,1,1),
  X(1,1,0,1,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,1,1,1),
  X(0,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 101 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 102 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,1,1,0,1,1),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(1,1,1,1,1,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 103 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,1),
  X(1,1,0,1,1,0),
  X(1,1,0,1,1,0),
  X(0,1,1,1,0,0),
  X(1,1,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  /* 6x13 bold character 104 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,1,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 105 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 106 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(0,0,0,1,1,0),
  X(1,1,0,1,1,0),
  X(1,1,0,1,1,0),
  X(0,1,1,1,0,0),
  /* 6x13 bold character 107 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,1,1,0),
  X(1,1,1,1,0,0),
  X(1,1,1,0,0,0),
  X(1,1,1,1,0,0),
  X(1,1,0,1,1,0),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 108 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 109 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 110 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,1,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 111 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 112 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,1,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,0,1,1),
  X(1,1,1,1,1,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  /* 6x13 bold character 113 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,1),
  X(1,1,0,1,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,1,1,1),
  X(0,1,1,1,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  X(0,0,0,0,1,1),
  /* 6x13 bold character 114 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,0),
  X(1,1,1,0,1,1),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 115 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(0,1,1,1,0,0),
  X(0,0,0,1,1,0),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 116 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(1,1,1,1,1,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,0,0),
  X(0,1,1,0,1,1),
  X(0,0,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 117 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,1,1,1),
  X(0,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 118 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 119 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(1,1,1,1,1,1),
  X(0,1,1,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 120 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,1,1,0),
  X(1,1,0,0,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 121 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,0,1,1),
  X(1,1,0,1,1,1),
  X(0,1,1,1,1,1),
  X(0,0,0,0,1,1),
  X(1,1,0,0,1,1),
  X(0,1,1,1,1,0),
  /* 6x13 bold character 122 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(1,1,0,0,0,0),
  X(1,1,1,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 123 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,1,1,1),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,1,1,1,0,0),
  X(0,0,0,1,1,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,1,1,1),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 124 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 125 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(1,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,1,0),
  X(0,1,1,0,0,0),
  X(0,0,1,1,0,0),
  X(0,0,1,1,0,0),
  X(1,1,1,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 126 */
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,1,1,0,1,0),
  X(1,1,1,1,1,0),
  X(1,0,0,1,1,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  X(0,0,0,0,0,0),
  /* 6x13 bold character 127 */
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
  X(0,1,0,1,0,0),
  X(1,0,1,0,1,0),
};

#undef X

/*****************************************************************************/

/* built-in 7x13 font */

/* This macro builds 7-bit wide rows. */
#define X(a, b, c, d, e, f, g) \
  ((a<<6)|(b<<5)|(c<<4)|(d<<3)|(e<<2)|(f<<1)|(g<<0))

static unsigned char font_7x13[256 * 13] = {
  /* 7x13 normal character 0 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 1 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 2 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,0),
  X(1,0,0,0,0,0,1),
  X(1,0,1,1,1,0,1),
  X(1,0,1,0,0,0,1),
  X(1,0,1,0,0,0,1),
  X(1,0,1,1,1,0,1),
  X(1,0,0,0,0,0,1),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 3 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 4 */
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  /* 7x13 normal character 5 */
  X(1,1,1,1,1,1,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,1,1,1,1,1,1),
  /* 7x13 normal character 6 */
  X(0,1,1,1,1,1,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,0),
  /* 7x13 normal character 7 */
  X(0,1,1,1,1,1,0),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(0,1,1,1,1,1,0),
  /* 7x13 normal character 8 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  /* 7x13 normal character 9 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,0,0,0,0,0,1),
  X(1,1,1,1,1,1,1),
  /* 7x13 normal character 10 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 11 */
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 12 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  /* 7x13 normal character 13 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  /* 7x13 normal character 14 */
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 15 */
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(1,1,1,1,1,1,1),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  /* 7x13 normal character 16 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 17 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 18 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 19 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 20 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 21 */
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  /* 7x13 normal character 22 */
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  /* 7x13 normal character 23 */
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 24 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  /* 7x13 normal character 25 */
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  /* 7x13 normal character 26 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,1,1,0,0),
  X(0,1,1,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 27 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,1,1,0,0),
  X(0,1,1,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 28 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,1,0,0,1,0,0),
  X(0,1,0,0,1,0,0),
  X(0,1,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 29 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,1,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 30 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,0,0),
  X(1,1,1,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,1,0),
  X(1,0,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 31 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 32 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 33 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 34 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,0,1,0,0),
  X(0,0,1,0,1,0,0),
  X(0,0,1,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 35 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,0,1,0,0),
  X(0,0,1,0,1,0,0),
  X(0,1,1,1,1,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,1,1,1,1,0),
  X(0,0,1,0,1,0,0),
  X(0,0,1,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 36 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,1,1,1,0),
  X(0,1,0,1,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,1,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 37 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,0,0,0,1,0),
  X(1,0,1,0,0,1,0),
  X(0,1,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,1,0,0,1,0,0),
  X(1,0,0,1,0,1,0),
  X(1,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 38 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(1,0,0,1,0,0,0),
  X(1,0,0,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(1,0,0,1,0,1,0),
  X(1,0,0,0,1,0,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 39 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 40 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 41 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 42 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,0,0,1,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 43 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 44 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 45 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 46 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 47 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 48 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,0,0,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 49 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 50 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 51 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 52 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 53 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,1,1,1,0,0),
  X(1,1,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 54 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,1,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,1,1,1,0,0),
  X(1,1,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 55 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 56 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 57 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,1,1,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 58 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 59 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 60 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 61 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 62 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 63 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 64 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,1,1,1,0),
  X(1,0,1,0,0,1,0),
  X(1,0,1,0,1,1,0),
  X(1,0,0,1,0,1,0),
  X(1,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 65 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,1,1,1,1,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 66 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 67 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 68 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,1,0),
  X(1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 69 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 70 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 71 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,1,1,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,1,1,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 72 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,1,1,1,1,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 73 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 74 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 75 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,1,0,0,0),
  X(1,0,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,0,1,0,0,0,0),
  X(1,0,0,1,0,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 76 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 77 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,0,1,1,0,1,0),
  X(1,0,1,1,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 78 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,1,0,0,0,1,0),
  X(1,0,1,0,0,1,0),
  X(1,0,0,1,0,1,0),
  X(1,0,0,0,1,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 79 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 80 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,1,1,1,1,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 81 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,1,0,0,1,0),
  X(1,0,0,1,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 82 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,1,1,1,1,0,0),
  X(1,0,1,0,0,0,0),
  X(1,0,0,1,0,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 83 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 84 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 85 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 86 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,0,0,1,0,0),
  X(0,1,0,0,1,0,0),
  X(0,1,0,0,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 87 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,1,1,0,1,0),
  X(1,0,1,1,0,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 88 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,0,0,1,0,0),
  X(0,1,0,0,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,1,0,0),
  X(0,1,0,0,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 89 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(0,1,0,1,0,0,0),
  X(0,1,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 90 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 91 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 92 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 93 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 94 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 95 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 96 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 97 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,1,0),
  X(0,1,1,1,1,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,1,1,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 98 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,1,1,1,0,0),
  X(1,1,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,1,0,0,0,1,0),
  X(1,0,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 99 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 100 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,1,1,1,0,1,0),
  X(1,0,0,0,1,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,1,1,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 101 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,1,1,1,1,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 102 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 103 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,1,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(0,1,1,1,0,0,0),
  X(1,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  /* 7x13 normal character 104 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,1,1,1,0,0),
  X(1,1,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 105 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 106 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(0,1,1,1,0,0,0),
  /* 7x13 normal character 107 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,1,0,0,0),
  X(1,1,1,0,0,0,0),
  X(1,0,0,1,0,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 108 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 109 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,1,0,0,0),
  X(1,0,1,0,1,0,0),
  X(1,0,1,0,1,0,0),
  X(1,0,1,0,1,0,0),
  X(1,0,1,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 110 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,1,1,1,0,0),
  X(1,1,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 111 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 112 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,1,1,1,0,0),
  X(1,1,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,1,0,0,0,1,0),
  X(1,0,1,1,1,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  /* 7x13 normal character 113 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,1,0),
  X(1,0,0,0,1,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,1,1,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,0,1,0),
  /* 7x13 normal character 114 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,1,1,1,0,0),
  X(0,1,0,0,0,1,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 115 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 116 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(0,1,0,0,0,1,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 117 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,1,1,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 118 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(0,1,0,1,0,0,0),
  X(0,1,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 119 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,0,0,1,0,0),
  X(1,0,1,0,1,0,0),
  X(1,0,1,0,1,0,0),
  X(1,0,1,0,1,0,0),
  X(0,1,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 120 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(0,1,0,0,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,1,0,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 121 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(1,0,0,0,1,1,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,1,0),
  X(1,0,0,0,0,1,0),
  X(0,1,1,1,1,0,0),
  /* 7x13 normal character 122 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,1,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 123 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0),
  X(0,0,0,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 124 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 125 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 126 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,0,0,1,0),
  X(0,1,0,1,0,1,0),
  X(0,1,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 normal character 127 */
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  /* 7x13 bold character 0 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 1 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 2 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,0),
  X(1,0,0,0,0,0,1),
  X(1,0,1,1,1,0,1),
  X(1,0,1,0,0,0,1),
  X(1,0,1,0,0,0,1),
  X(1,0,1,1,1,0,1),
  X(1,0,0,0,0,0,1),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 3 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 4 */
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  /* 7x13 bold character 5 */
  X(1,1,1,1,1,1,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,1,1,1,1,1,1),
  /* 7x13 bold character 6 */
  X(0,1,1,1,1,1,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,0),
  /* 7x13 bold character 7 */
  X(0,1,1,1,1,1,0),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(1,0,0,0,0,0,1),
  X(0,1,1,1,1,1,0),
  /* 7x13 bold character 8 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  /* 7x13 bold character 9 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,0,0,0,0,0,1),
  X(1,1,1,1,1,1,1),
  /* 7x13 bold character 10 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 11 */
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 12 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  /* 7x13 bold character 13 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1),
  X(0,0,1,1,1,1,1),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  /* 7x13 bold character 14 */
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,1,1,1),
  X(0,0,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 15 */
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  /* 7x13 bold character 16 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 17 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 18 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 19 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 20 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 21 */
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,1,1,1),
  X(0,0,1,1,1,1,1),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  /* 7x13 bold character 22 */
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  /* 7x13 bold character 23 */
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 24 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  /* 7x13 bold character 25 */
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  /* 7x13 bold character 26 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 27 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 28 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,1,1,0,1,1,0),
  X(0,1,1,0,1,1,0),
  X(0,1,1,0,1,1,0),
  X(1,1,1,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 29 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0),
  X(0,0,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 30 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,1,1,0,1,1,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,1,1,0),
  X(1,0,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 31 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 32 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 33 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 34 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,0,1,1,0),
  X(0,1,1,0,1,1,0),
  X(0,1,1,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 35 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,0,1,0,0),
  X(0,0,1,0,1,0,0),
  X(0,1,1,1,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,1,1,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,1,0,1,0,0),
  X(0,0,1,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 36 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,1,1,0,1,0),
  X(1,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,1,0),
  X(1,0,1,1,0,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 37 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,0,0,1,0),
  X(1,0,1,0,1,1,0),
  X(1,1,1,0,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,0,1,1,1,0),
  X(1,1,0,1,0,1,0),
  X(1,0,0,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 38 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,0,0),
  X(1,1,0,1,1,0,0),
  X(1,1,0,1,1,0,0),
  X(1,1,0,1,1,0,0),
  X(0,1,1,1,0,0,0),
  X(1,1,0,1,0,1,0),
  X(1,1,0,1,1,1,0),
  X(1,1,0,1,1,0,0),
  X(0,1,1,1,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 39 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 40 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 41 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 42 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,0,0,1,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 43 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 44 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 45 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 46 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 47 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 48 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,0,0,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,0,0,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 49 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,1,0,0,0),
  X(1,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 50 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,1,1,1,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 51 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 52 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,1,0),
  X(0,0,1,1,1,1,0),
  X(0,1,1,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 53 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 54 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 55 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 56 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 57 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 58 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 59 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 60 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 61 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 62 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 63 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 64 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,0,0,0,1,1,0),
  X(1,0,0,0,1,1,0),
  X(1,0,1,1,1,1,0),
  X(1,0,1,0,1,1,0),
  X(1,0,1,1,1,1,0),
  X(1,0,0,0,0,0,0),
  X(1,0,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 65 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 66 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 67 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 68 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 69 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 70 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 71 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 72 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 73 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 74 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 75 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,1,1,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,1,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,0,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 76 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 77 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 78 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,0,1,1,0),
  X(1,1,1,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,1,1,1,0),
  X(1,1,0,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 79 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 80 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 81 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,0,1,1,0),
  X(1,1,0,1,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 82 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,0,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 83 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 84 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 85 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 86 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,0,0,1,0,0),
  X(0,1,0,0,1,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 87 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 88 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,0,0,0,0,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,0,0,1,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,1,0,0,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 89 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 90 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 91 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 92 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 93 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 94 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 95 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 96 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 97 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 98 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 99 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 100 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 101 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 102 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,1,1,0,1,1,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 103 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  /* 7x13 bold character 104 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 105 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 106 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  /* 7x13 bold character 107 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,1,1,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,0,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 108 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 109 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,1,1,0,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 110 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 111 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 112 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  /* 7x13 bold character 113 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,0,1,1,0),
  /* 7x13 bold character 114 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 115 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 116 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 117 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 118 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 119 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,0),
  X(0,1,0,0,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 120 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  X(0,1,1,1,1,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 121 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,1,0),
  X(0,0,0,0,1,1,0),
  X(1,1,0,0,1,1,0),
  X(0,1,1,1,1,0,0),
  /* 7x13 bold character 122 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,1,1,0),
  X(0,0,0,1,1,0,0),
  X(0,1,1,0,0,0,0),
  X(1,1,0,0,0,0,0),
  X(1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 123 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0),
  X(0,0,1,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 124 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 125 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0),
  X(0,0,0,1,1,0,0),
  X(0,1,1,1,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 126 */
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,1,1,0,0,1,0),
  X(1,1,1,1,1,1,0),
  X(1,0,0,1,1,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0),
  /* 7x13 bold character 127 */
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0),
  X(0,0,1,0,1,0,0),
  X(0,1,0,1,0,1,0)
};

#undef X

/*****************************************************************************/

/* built-in 9x15 font */

/* This macro builds 9-bit wide rows. */
#define X(a, b, c, d, e, f, g, h, i) \
  ((a<<8)|(b<<7)|(c<<6)|(d<<5)|(e<<4)|(f<<3)|(g<<2)|(h<<1)|(i<<0))

static unsigned short font_9x15[256 * 15] = {
  /* 9x15 normal character 0 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 1 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 2 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,0,0,0,0,0,0,1),
  X(0,1,0,0,1,1,0,0,1),
  X(0,1,0,1,0,0,0,0,1),
  X(0,1,0,1,0,0,0,0,1),
  X(0,1,0,0,1,1,0,0,1),
  X(0,1,0,0,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 3 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 4 */
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  /* 9x15 normal character 5 */
  X(1,1,1,1,1,1,1,1,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,1,1,1,1,1,1,1,1),
  /* 9x15 normal character 6 */
  X(0,0,1,1,1,1,1,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  /* 9x15 normal character 7 */
  X(0,0,1,1,1,1,1,0,0),
  X(0,1,0,0,0,0,0,1,0),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(0,1,0,0,0,0,0,1,0),
  X(0,0,1,1,1,1,1,0,0),
  /* 9x15 normal character 8 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  /* 9x15 normal character 9 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,1,1,1,1,1,1,1,1),
  /* 9x15 normal character 10 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 11 */
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 12 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  /* 9x15 normal character 13 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,1,1,1),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  /* 9x15 normal character 14 */
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 15 */
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  /* 9x15 normal character 16 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 17 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 18 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 19 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 20 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 21 */
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,1,1,1),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  /* 9x15 normal character 22 */
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  /* 9x15 normal character 23 */
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 24 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  /* 9x15 normal character 25 */
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  /* 9x15 normal character 26 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 27 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 28 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,1,0,0,1,0),
  X(0,0,0,0,1,0,0,1,0),
  X(0,0,0,0,1,0,0,1,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 29 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 30 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,0,1,0,0,0,1),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,1,0,1,0,0,1,1),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 31 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 32 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 33 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 34 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0,1,0),
  X(0,0,0,0,1,0,0,1,0),
  X(0,0,0,0,1,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 35 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,1,0,0),
  X(0,0,0,1,0,0,1,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,1,0,0,1,0,0),
  X(0,0,0,1,0,0,1,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,1,0,0,1,0,0),
  X(0,0,0,1,0,0,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 36 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,1,0,0,0),
  X(0,0,1,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,1,0,0,1),
  X(0,0,0,0,0,1,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 37 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,1,0,1,0,0,1,0),
  X(0,0,1,0,1,0,0,1,0),
  X(0,0,0,1,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,1,0),
  X(0,0,0,1,0,0,1,0,1),
  X(0,0,0,1,0,0,1,0,1),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 38 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,1,0,0,0,1,0,0),
  X(0,0,1,0,0,0,1,0,0),
  X(0,0,1,0,0,0,1,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,1,0,0,0,1,0,0),
  X(0,0,1,0,0,0,1,0,1),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,1,1,0),
  X(0,0,0,1,1,1,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 39 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 40 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 41 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 42 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 43 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 44 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 45 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 46 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 47 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 48 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 49 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 50 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 51 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 52 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,0,1,0),
  X(0,0,0,0,1,0,0,1,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 53 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,1,1,1,1,0),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 54 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,1,1,1,1,0),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 55 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 56 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 57 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,1,1),
  X(0,0,0,1,1,1,1,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 58 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 59 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 60 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 61 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 62 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 63 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 64 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,1,1,1,1),
  X(0,0,1,0,1,0,0,0,1),
  X(0,0,1,0,1,0,0,1,1),
  X(0,0,1,0,0,1,1,0,1),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 65 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 66 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 67 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 68 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 69 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 70 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 71 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,1,1,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 72 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 73 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 74 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,1,1,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 75 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,1,0,0),
  X(0,0,1,0,0,1,0,0,0),
  X(0,0,1,1,1,0,0,0,0),
  X(0,0,1,0,1,0,0,0,0),
  X(0,0,1,0,0,1,0,0,0),
  X(0,0,1,0,0,0,1,0,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 76 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 77 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,0,1,0,1,0,1),
  X(0,0,1,0,1,0,1,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 78 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,1,0,1,0,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,0,1,0,1),
  X(0,0,1,0,0,0,0,1,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 79 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 80 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 81 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,0,1,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 82 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,1,0,0,1,0,0,0),
  X(0,0,1,0,0,0,1,0,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 83 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 84 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 85 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 86 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 87 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,1,0,1,0,1),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 88 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 89 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 90 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 91 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 92 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 93 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 94 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 95 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 96 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 97 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,1,1),
  X(0,0,0,1,1,1,1,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 98 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,1,1,1,1,0),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,1,0,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 99 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 100 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,0,1),
  X(0,0,1,0,0,0,0,1,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,1,1),
  X(0,0,0,1,1,1,1,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 101 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 102 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,0,1,0,0,0,1),
  X(0,0,0,0,1,0,0,0,1),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 103 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,0,1),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  /* 9x15 normal character 104 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,1,1,1,1,0),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 105 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 106 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,0,1,1,1,1,0,0),
  /* 9x15 normal character 107 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,1,1,0),
  X(0,0,1,0,1,1,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,0,1,1,0,0,0),
  X(0,0,1,0,0,0,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 108 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 109 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,1,1,0),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 110 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,1,1,1,1,0),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 111 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 112 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,1,1,1,1,0),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,1,0,0,0,0,1),
  X(0,0,1,0,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,0),
  /* 9x15 normal character 113 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,0,1),
  X(0,0,1,0,0,0,0,1,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,1,1),
  X(0,0,0,1,1,1,1,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,1),
  /* 9x15 normal character 114 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,1,1,1,0),
  X(0,0,0,1,1,0,0,0,1),
  X(0,0,0,1,0,0,0,0,1),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 115 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 116 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,1,0,0,0,1),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 117 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,0,1,1,1,1,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 118 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 119 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,1,0,1,0,1),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 120 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,1,0,0),
  X(0,0,0,1,0,0,0,1,0),
  X(0,0,1,0,0,0,0,0,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 121 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,1,0,0,0,1,1,0),
  X(0,0,0,1,1,1,0,1,0),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,1,0,0,0,0,1,0),
  X(0,0,0,1,1,1,1,0,0),
  /* 9x15 normal character 122 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,1,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,1,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 123 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,1,1,1),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,1,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 124 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 125 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,1,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 126 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,0,0,0,1),
  X(0,0,1,0,0,1,0,0,1),
  X(0,0,1,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 normal character 127 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 0 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 1 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 2 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(1,1,0,0,0,0,0,1,1),
  X(1,1,0,0,1,1,0,1,1),
  X(1,1,0,1,0,0,0,1,1),
  X(1,1,0,1,0,0,0,1,1),
  X(1,1,0,0,1,1,0,1,1),
  X(1,1,0,0,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 3 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 4 */
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  /* 9x15 bold character 5 */
  X(1,1,1,1,1,1,1,1,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,1,1,1,1,1,1,1,1),
  /* 9x15 bold character 6 */
  X(0,0,1,1,1,1,1,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  /* 9x15 bold character 7 */
  X(0,0,1,1,1,1,1,0,0),
  X(0,1,0,0,0,0,0,1,0),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,0,0,0,0,0,0,0,1),
  X(0,1,0,0,0,0,0,1,0),
  X(0,0,1,1,1,1,1,0,0),
  /* 9x15 bold character 8 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  /* 9x15 bold character 9 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,0,0,0,0,0,0,0,1),
  X(1,1,1,1,1,1,1,1,1),
  /* 9x15 bold character 10 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 11 */
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 12 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  /* 9x15 bold character 13 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,1,1),
  X(0,0,0,0,1,1,1,1,1),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  /* 9x15 bold character 14 */
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,1,1,1),
  X(0,0,0,0,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 15 */
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  /* 9x15 bold character 16 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 17 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 18 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 19 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 20 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 21 */
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,1,1,1),
  X(0,0,0,0,1,1,1,1,1),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  /* 9x15 bold character 22 */
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(1,1,1,1,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  /* 9x15 bold character 23 */
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 24 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  /* 9x15 bold character 25 */
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  /* 9x15 bold character 26 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,1,1,1),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,1,1,1,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,0,0,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 27 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,0,0,0,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,1,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 28 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,1,0,0,1,1,0),
  X(0,0,0,1,0,0,1,1,0),
  X(0,0,0,1,0,0,1,1,0),
  X(0,1,1,1,0,0,1,1,0),
  X(0,1,1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 29 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,1,1,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 30 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,1,1,0,0,1,1),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,1,1,1,1,1,0,0,0),
  X(0,1,0,1,1,1,1,1,1),
  X(0,1,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 31 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 32 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 33 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 34 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,0,1,1,0),
  X(0,0,0,1,1,0,1,1,0),
  X(0,0,0,1,1,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 35 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,0,1,1,0,0),
  X(0,0,1,1,0,1,1,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,1,1,0,1,1,0,0),
  X(0,0,1,1,0,1,1,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,1,1,0,1,1,0,0),
  X(0,0,1,1,0,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 36 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,1,0,1,0,1,1),
  X(0,0,1,1,0,1,0,0,0),
  X(0,0,1,1,1,1,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,1,1,1,1),
  X(0,0,0,0,0,1,0,1,1),
  X(0,0,1,1,0,1,0,1,1),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 37 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0,1,1),
  X(0,1,1,1,1,0,1,1,0),
  X(0,1,1,1,1,0,1,1,0),
  X(0,0,1,1,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,1,1,0),
  X(0,0,1,1,0,1,1,1,1),
  X(0,0,1,1,0,1,1,1,1),
  X(0,1,1,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 38 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,0,0,0),
  X(0,1,1,0,0,1,1,0,0),
  X(0,1,1,0,0,1,1,0,0),
  X(0,1,1,0,0,1,1,0,0),
  X(0,0,1,1,1,1,0,0,0),
  X(0,1,1,0,0,1,1,0,0),
  X(0,1,1,0,0,1,1,1,1),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,1,1,1,0),
  X(0,0,1,1,1,1,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 39 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 40 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 41 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 42 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,1,1,0,1,0,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,1,1,1,0,1,1,1,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,1,1,0,1,0,1,1,0),
  X(0,0,0,0,1,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 43 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 44 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 45 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 46 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 47 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 48 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 49 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,1,1,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 50 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 51 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 52 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,1,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,1,1,0,1,1,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 53 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,1,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 54 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,1,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 55 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 56 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 57 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,1,1,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 58 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 59 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 60 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 61 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 62 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 63 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 64 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,1,1,1,1,1),
  X(0,1,1,1,1,0,0,1,1),
  X(0,1,1,1,1,0,1,1,1),
  X(0,1,1,0,1,1,1,1,1),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 65 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 66 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 67 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 68 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 69 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 70 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 71 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,1,1,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 72 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 73 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 74 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 75 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,1,1,0,0),
  X(0,1,1,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0,0,0),
  X(0,1,1,1,1,0,0,0,0),
  X(0,1,1,0,1,1,0,0,0),
  X(0,1,1,0,0,1,1,0,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 76 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 77 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,0,0,1,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 78 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,0,0,0,1,1),
  X(0,1,1,1,1,0,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,0,1,1,1,1),
  X(0,1,1,0,0,0,1,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 79 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 80 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 81 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,0,1,1,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 82 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,0,1,1,0,0,0),
  X(0,1,1,0,0,1,1,0,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 83 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 84 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 85 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 86 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 87 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 88 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 89 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 90 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 91 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 92 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 93 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 94 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 95 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(1,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 96 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,0,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 97 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,1,1,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 98 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,1,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 99 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 100 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,1,1,0,0,0,1,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,1,1,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 101 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 102 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,1,1,0,0,1,1),
  X(0,0,0,1,1,0,0,1,1),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,1,1,1,1,1,1,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 103 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  /* 9x15 bold character 104 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,1,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 105 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 106 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  /* 9x15 bold character 107 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,1,1,1,0),
  X(0,1,1,1,1,1,0,0,0),
  X(0,1,1,1,0,0,0,0,0),
  X(0,1,1,1,1,1,0,0,0),
  X(0,1,1,0,0,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 108 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 109 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 110 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,1,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 111 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 112 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,1,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,1,0,0,0,1,1),
  X(0,1,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,0,0),
  /* 9x15 bold character 113 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,1,1,0,0,0,1,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,1,1,1),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,1,1),
  /* 9x15 bold character 114 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,1,1,1,1,0),
  X(0,0,1,1,1,0,0,1,1),
  X(0,0,1,1,0,0,0,1,1),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 115 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,0,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 116 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,1,1,1,1,1,1,1,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,1,1,0,0,1,1),
  X(0,0,0,0,1,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 117 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,0,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 118 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 119 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,0,0,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 120 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,1,1,0,0,1,1,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,1,1,0,0),
  X(0,0,1,1,0,0,1,1,0),
  X(0,1,1,0,0,0,0,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 121 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,1,1,0,0,1,1,1,0),
  X(0,0,1,1,1,1,1,1,0),
  X(0,0,0,0,0,0,1,1,0),
  X(0,1,1,0,0,0,1,1,0),
  X(0,0,1,1,1,1,1,0,0),
  /* 9x15 bold character 122 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,1,1,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,1,1,0,0,0,0,0),
  X(0,1,1,1,1,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 123 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,1,1,1,1),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,0,1,1,1,0,0,0),
  X(0,0,0,0,0,1,1,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,1,1,1,1),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 124 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 125 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,1,1,1,1,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,0,1,1,1,0,0),
  X(0,0,0,1,1,0,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,0,0,0,1,1,0,0,0),
  X(0,1,1,1,1,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 126 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,1,1,1,0,0,1,1),
  X(0,1,1,0,1,1,0,1,1),
  X(0,1,1,0,0,1,1,1,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  /* 9x15 bold character 127 */
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0),
  X(0,0,0,0,0,0,0,0,0)
};

#undef X

/*****************************************************************************/

/* The terminal emulator module is organized:
 *   addstr()            -- append a string, advance pointer
 *   addnum()            -- append a number, advance pointer
 *   termcap()           -- answerback with termcap entry
 *   bbits()             -- compute border bits
 *   border()            -- draw a horizontal border line
 *   cursor()            -- draw the cursor, toggle between on and off
 *   space()             -- draw a space at the specified location
 *   draw()              -- draw a character at the specified location
 *   clear()             -- clear one or more lines
 *   scroll()            -- scroll lines up or down onto specified line
 *   blink()             -- blink the display, a visual bell
 *   setup()             -- setup display context
 *   font()              -- setup the font
 *   terminal_emulator() -- run terminal emulator to display a character
 *   terminal_buffer()   -- switch to a new display buffer
 *   terminal_setup()    -- prepare a terminal context
 */

/* This is the termcap specification for the Mach Terminal Emulator. */
static char mach_termcap[] =
  "mt|mach:"       /* term = "mach" */
  "do=^J:"         /* cursor down */
  "le=^H:"         /* cursor left */
  "bs:"            /* can backspace with ^H */
  "cd=\\EJ:"       /* clear to end of display */
  "ce=\\EK:"       /* clear to end of line */
  "cl=\\EH\\EJ:"   /* clear screen and home cursor */
  "cm=\\EY%+ %+ :" /* screen-relative cursor motion */
  "nd=\\EC:"       /* non-destrictive space, cursor right */
  "pt:"            /* has hardware tabs */
  "sr=\\EI:"       /* scroll text down */
  "up=\\EA:"       /* move cursor up */
  "ku=\\E[A:"      /* sent by up arrow key */
  "kd=\\E[B:"      /* sent by down arrow key */
  "kr=\\E[C:"      /* sent by right arrow key */
  "kl=\\E[D:"      /* sent by left arrow key */
  "kb=\\177:"      /* sent by backspace key */
  "kD=\\177:"      /* sent by delete key */
  "so=\\Ej:"       /* begin standout mode */
  "se=\\Ek:"       /* end standout mode */
  "us=\\El:"       /* start underscore mode */
  "ue=\\Em:"       /* end underscore mode */
  "as=\\En:"       /* start alternate character set */
  "ae=\\Eo:"       /* end alternate character set */
  "md=\\Ep:"       /* turn on bold attribute */
  "me=\\Eq:"       /* turn off all attributes */
  "bl=^G"          /* bell */
  "vb=^F"          /* visable bell */
  "hs:"            /* has extra status line */
  "es:"            /* escape can be used on status line */
  "ts=\\ES%+ :"    /* go to status line, column n */
  "fs=^B:";        /* return from status line */
  /* co */         /* number of columns in a line */
  /* li */         /* number of lines on screen */
  /* ws */         /* number of columns in status line */
  /* co, li and ws are computed by termcap() */

/* The following codes are available in addition to those defined in the
 * termcap entry:
 * ^A  -- enter and clear the status line
 * ^B  -- exit the status line
 * ^C  -- return the termcap entry
 * ^D  -- toggle between white and black
 * ^E% -- set cursor graphics character
 */

/*****************************************************************************/

/* convert a "normal" character to a "graphics" character */
/* for example, TOGRAPHIC('a') is the copyright symbol */
#define TOGRAPHIC(c) (c - 95)

/* bits and bytes */
#define INTBYTES   4
#define SHORTBYTES 2
#define INTBITS   32
#define SHORTBITS 16
#define CHARBITS   8

/* black on white displays have a neat little border... */
#define VBORDER 4 /* vertical border width, in pixels */
#define HBORDER 3 /* horizontal border width, in pixels */

/* mode bits */
#define BOLD      1
#define UNDERLINE 2
#define INVERSE   4
#define GRAPHIC   8

/* this is the actual terminal context structure */
typedef struct context {
  unsigned int *buffer;        /* bit-mapped display, one bit pixels */
  int height;                  /* display height, in pixels */
  int width;                   /* display width, in pixels */
  int rowints;                 /* display buffer width, in int's */
  int rowsize;                 /* display width, in int's */
  unsigned int left;           /* left border bits */
  unsigned int right1, right2; /* right border bits */
  int tmargin;                 /* y offset for centered display */
  int rmargin;                 /* the first pixel of the right border */
  int lmargin;                 /* x offset for centered display */
  char *font;                  /* the font bits */
  int font_height;             /* font height, in pixels */
  int font_width;              /* font width, in pixels */
  unsigned int mask;           /* font mask bits */
  char cursor;                 /* graphics character to use for a cursor */
  int white_on_black;          /* white on black or black on white */
  void (*ungetc)(char, char*); /* input write function */
  char *ungetc_argument;       /* 2nd argument to ungetc function */
  void (*bell)(void);          /* beep the bell */
  int max_x, max_y;            /* display size, in characters */
  int cursor_x, cursor_y;      /* current cursor position */
  int new_x, new_y;            /* new cursor position */
  int state;                   /* terminal emulator state */
  int mode;                    /* display attribute mode */
  int status;                  /* status line flag, x + 1 */
  int status_mode;             /* status line attribute mode */
  int am_count;                /* automatic margin counter */
} *context_t;

/* convert character coordinates to pixel coordinates */
#define TOPIXEL(t, x, y) \
  if (y == t->max_y) \
    y = (y * t->font_height) + t->tmargin + HBORDER; \
  else y = (y * t->font_height) + t->tmargin; \
  x = (x * t->font_width) + t->lmargin

/* buffer index to first row on text line */
/* y is a character coordinate */
#define ROWINDEX(t, y) ((((y) * t->font_height) + t->tmargin) * t->rowints)

/* buffer index to first row on status line */
#define STATUSINDEX(t) \
  (((t->max_y * t->font_height) + t->tmargin + HBORDER) * t->rowints)

/* buffer index to int that contains the specified pixel */
/* x and y are pixel coordinates */
#define INDEX(t, x, y) ((y * t->rowints) + (x / INTBITS))

/* determine pixel shift for a given font width */
/* if this is negative, then character is split over two int's */
#define SHIFT(t, x, width) (INTBITS - width - x % INTBITS)

/* compute white on black bits with a left shift */
#define LWBCHAR(b, mask, bits, shift) (b & ~mask) | (~(bits << shift) & mask)

/* compute white on black bits with a right shift */
#define RWBCHAR(b, mask, bits, shift) (b & ~mask) | (~(bits >> shift) & mask)

/* compute black on white bits with a left shift */
#define LBWCHAR(b, mask, bits, shift) (b & ~mask) | (bits << shift)

/* compute black on white bits with a right shift */
#define RBWCHAR(b, mask, bits, shift) (b & ~mask) | (bits >> shift)

/* compute cursor bits with a left shift */
#define LCURSOR(b, mask, bits, shift) (b ^ (bits << shift))

/* compute cursor bits with a right shift */
#define RCURSOR(b, mask, bits, shift) (b ^ (bits >> shift))

/* this is the exported terminal context structure */
typedef char terminal_context_t[sizeof(struct context)];

/* this context is used when a zero pointer is given */
static struct context default_context;

/*****************************************************************************/

/* append a string, advance pointer */
static void addstr(register char **p, register char *b)
{
  while (*b) *(*p)++ = *b++;
}

/* append a number, advance pointer */
static void addnum(register char **p, register int n, register int maxw)
{
  char buffer[30];
  register char *b = &buffer[sizeof(buffer) - 1];

  *(b--) = 0;
  do {
    *b-- = '0' + n % 10;
    n /= 10;
  } while ((--maxw > 0) && (n != 0));
  b++;
  addstr(p, b);
}

/* answerback with termcap entry */
static void termcap(register context_t t)
{
  char buffer[sizeof(mach_termcap) + 23], *p;

  /* if no input write function, do nothing */
  if (!t->ungetc) return;

  /* compose termcap entry */
  *(p = buffer) = 0;
  addstr(&p, mach_termcap);
  addstr(&p, "co#"); addnum(&p, t->max_x, 3);
  addstr(&p, ":li#"); addnum(&p, t->max_y, 3);
  addstr(&p, ":ws#"); addnum(&p, t->max_x, 3);
  addstr(&p, ":\015");
  *p = 0;

  /* write termcap entry to input stream */
  for (p = buffer; *p; p++) (*t->ungetc)(*p, t->ungetc_argument);

} /* termcap() */

/*****************************************************************************/

/* compute border bits */
static void bbits(register context_t t,
                  char *bits,
                  unsigned int *left,
                  unsigned int *right1,
                  unsigned int *right2,
                  unsigned int *filler)
{
  unsigned int left_bits, right_bits;
  int i, shift;

  /* compute filler bits */
  *filler = (bits[VBORDER] == '0') ? 0x00000000 : 0xFFFFFFFF;

  /* compute left and right border bits */
  left_bits = 0xFFFFFFFF;
  right_bits = *filler;
  for (i = 0; i < VBORDER; i++) {
    left_bits = (left_bits << 1) | ((bits[i] == '0') ? 0 : 1);
    right_bits =
      (right_bits << 1) | ((bits[VBORDER - i - 1] == '0') ? 0 : 1);
  }

  /* compute left border */
  shift = INTBITS - t->lmargin;
  *left = (left_bits << shift) | (*filler >> (INTBITS - shift));

  /* compute right border */
  if ((shift = SHIFT(t, t->rmargin, VBORDER)) < 0) {
    shift = - shift;
    *right1 = (right_bits >> shift) | (*filler << (INTBITS - shift));
    *right2 = (right_bits << (INTBITS - shift)) | (0xFFFFFFFF >> shift);
  }
  else {
    *right1 = (right_bits << shift) | (0xFFFFFFFF >> (INTBITS - shift));
    *right2 = 0;
  }

} /* bbits() */

/* draw a horizontal border line */
static void border(context_t t, int y, char *bits)
{
  unsigned int left, right1, right2, filler;
  register unsigned int *b;
  register int x;

  bbits(t, bits, &left, &right1, &right2, &filler);
  b = &t->buffer[y * t->rowints];
  *b++ = left;
  if (right2) {
    for (x = t->rowsize - 3; x--;) *b++ = filler;
    *b++ = right1;
    *b++ = right2;
  }
  else {
    for (x = t->rowsize - 2; x--;) *b++ = filler;
    *b++ = right1;
  }

} /* border() */

/*****************************************************************************/

/* draw the cursor, toggle between on and off */
static void cursor(register context_t t)
{
  register unsigned int *b, mask1, mask2;
  register unsigned char *bits1;
  register unsigned short *bits2;
  register int shift1, shift2, i, x, y;

  /* get pointer to font bits */
  if (t->font_width > CHARBITS)
    bits2 = &((unsigned short *)t->font)[t->cursor * t->font_height];
  else bits1 = &((unsigned char *)t->font)[t->cursor * t->font_height];

  /* convert to pixel coordinates */
  x = t->cursor_x; y = t->cursor_y;
  TOPIXEL(t, x, y);

  /* get pointer to pixel's int */
  b = &t->buffer[INDEX(t, x, y)];

  /* character spans two adjacent int's */
  if ((shift1 = SHIFT(t, x, t->font_width)) < 0) {
    mask1 = t->mask >> (shift1 = -shift1);
    mask2 = t->mask << (shift2 = INTBITS - shift1);

    /* font bits contained in two bytes */
    if (t->font_width > CHARBITS) {
      for (i = t->font_height - 1; i; i--, b += t->rowints, bits2++) {
        *b = RCURSOR(*b, mask1, *bits2, shift1);
        b[1] = LCURSOR(b[1], mask2, *bits2, shift2);
      }
      *b++ = RCURSOR(*b, mask1, *bits2, shift1);
      *b = LCURSOR(*b, mask2, *bits2, shift2);
    }

    /* font bits contained in one byte */
    else {
      for (i = t->font_height - 1; i; i--, b += t->rowints, bits1++) {
        *b = RCURSOR(*b, mask1, *bits1, shift1);
        b[1] = LCURSOR(b[1], mask2, *bits1, shift2);
      }
      *b++ = RCURSOR(*b, mask1, *bits1, shift1);
      *b = LCURSOR(*b, mask2, *bits1, shift2);
    }

  }

  /* character contained in one word */
  else {
    mask1 = t->mask << shift1;

    /* font bits contained in two bytes */
    if (t->font_width > CHARBITS) {
      for (i = t->font_height - 1; i; i--, b += t->rowints, bits2++)
        *b = LCURSOR(*b, mask1, *bits2, shift1);
      *b = LCURSOR(*b, mask1, *bits2, shift1);
    }

    /* font bits contained in one byte */
    else {
      for (i = t->font_height - 1; i; i--, b += t->rowints, bits1++)
        *b = LCURSOR(*b, mask1, *bits1, shift1);
      *b = LCURSOR(*b, mask1, *bits1, shift1);
    }

  }

} /* cursor() */

/*****************************************************************************/

/* draw a space at the specified location */
static void space(register context_t t, register int x, register int y)
{
  register unsigned int *b, space1, space2;
  register int shift, i;
  int normal;

  /* get display or status line mode */
  if (y == t->max_y) normal = t->status_mode & INVERSE;
  else if (t->white_on_black) normal = t->mode & INVERSE;
  else normal = !(t->mode & INVERSE);

  /* convert to pixel coordinates */
  TOPIXEL(t, x, y);

  /* get pointer to pixel's int */
  b = &t->buffer[INDEX(t, x, y)];

  /* character spans two adjacent int's */
  if ((shift = SHIFT(t, x, t->font_width)) < 0) {
    space1 = t->mask >> (-shift);
    space2 = t->mask << (INTBITS + shift);

    /* normal -- black on white space */
    if (normal) {
      for (i = t->font_height - 1; i; i--, b += t->rowints) {
        *b &= ~space1;
        b[1] &= ~space2; 
      }
      *b++ &= ~space1;
      *b &= ~space2;
    }

    /* inverse -- white on black space */
    else {
      for (i = t->font_height - 1; i; i--, b += t->rowints) {
        *b |= space1;
        b[1] |= space2;
      }
      *b++ |= space1;
      *b |= space2;
    }

  }

  /* character contained in one word */
  else {
    space1 = t->mask << shift;

    /* normal -- black on white space */
    if (normal) {
      for (i = t->font_height - 1; i; i--, b += t->rowints) *b &= ~space1;
      *b &= ~space1;
    }

    /* inverse -- white on black space */
    else {
      for (i = t->font_height - 1; i; i--, b += t->rowints) *b |= space1;
      *b |= space1;
    }

  }

} /* space() */

/*****************************************************************************/

/* draw a character */
static void draw(register context_t t, register int x, register int y, char c)
{
  register unsigned int *b, mask1, mask2;
  register unsigned char *bits1;
  register unsigned short *bits2;
  register int shift1, shift2, i;
  unsigned int mode;

  /* get display or status line mode */
  if (y == t->max_y) mode = t->status_mode ^ INVERSE;
  else if (t->white_on_black) mode = t->mode ^ INVERSE;
  else mode = t->mode;

  /* get pointer to font bits */
  if (mode & GRAPHIC) c = TOGRAPHIC(c);
  if (t->font_width > CHARBITS) {
    bits2 = &((unsigned short *)t->font)[c * t->font_height];
    if (mode & BOLD) bits2 = &bits2[128 * t->font_height];
  }
  else {
    bits1 = &((unsigned char *)t->font)[c * t->font_height];
    if (mode & BOLD) bits1 = &bits1[128 * t->font_height];
  }

  /* convert to pixel coordinates */
  TOPIXEL(t, x, y);

  /* get pointer to pixel's int */
  b = &t->buffer[INDEX(t, x, y)];

  /* character spans two adjacent int's */
  if ((shift1 = SHIFT(t, x, t->font_width)) < 0) {
    mask1 = t->mask >> (shift1 = -shift1);
    mask2 = t->mask << (shift2 = INTBITS - shift1);

    /* display white on black */
    if (mode & INVERSE) {

      /* font bits contained in two bytes */
      if (t->font_width > CHARBITS) {
        for (i = t->font_height - 1; i; i--, b += t->rowints, bits2++) {
          *b = RWBCHAR(*b, mask1, *bits2, shift1);
          b[1] = LWBCHAR(b[1], mask2, *bits2, shift2);
        }
        if (mode & UNDERLINE) {
          *b++ = RWBCHAR(*b, mask1, t->mask, shift1);
          *b = LWBCHAR(*b, mask2, t->mask, shift2);
        }
        else {
          *b++ = RWBCHAR(*b, mask1, *bits2, shift1);
          *b = LWBCHAR(*b, mask2, *bits2, shift2);
        }
      }

      /* font bits contained in one byte */
      else {
        for (i = t->font_height - 1; i; i--, b += t->rowints, bits1++) {
          *b = RWBCHAR(*b, mask1, *bits1, shift1);
          b[1] = LWBCHAR(b[1], mask2, *bits1, shift2);
        }
        if (mode & UNDERLINE) {
          *b++ = RWBCHAR(*b, mask1, t->mask, shift1);
          *b = LWBCHAR(*b, mask2, t->mask, shift2);
        }
        else {
          *b++ = RWBCHAR(*b, mask1, *bits1, shift1);
          *b = LWBCHAR(*b, mask2, *bits1, shift2);
        }
      }
    }

    /* display black on white */
    else {

      /* font bits contained in two bytes */
      if (t->font_width > CHARBITS) {
        for (i = t->font_height - 1; i; i--, b += t->rowints, bits2++) {
          *b = RBWCHAR(*b, mask1, *bits2, shift1);
          b[1] = LBWCHAR(b[1], mask2, *bits2, shift2);
        }
        if (mode & UNDERLINE) {
          *b++ = RBWCHAR(*b, mask1, t->mask, shift1);
          *b = LBWCHAR(*b, mask2, t->mask, shift2);
        }
        else {
          *b++ = RBWCHAR(*b, mask1, *bits2, shift1);
          *b = LBWCHAR(*b, mask2, *bits2, shift2);
        }
      }

      /* font bits contained in one byte */
      else {
        for (i = t->font_height - 1; i; i--, b += t->rowints, bits1++) {
          *b = RBWCHAR(*b, mask1, *bits1, shift1);
          b[1] = LBWCHAR(b[1], mask2, *bits1, shift2);
        }
        if (mode & UNDERLINE) {
          *b++ = RBWCHAR(*b, mask1, t->mask, shift1);
          *b = LBWCHAR(*b, mask2, t->mask, shift2);
        }
        else {
          *b++ = RBWCHAR(*b, mask1, *bits1, shift1);
          *b = LBWCHAR(*b, mask2, *bits1, shift2);
        }
      }
    }
  }

  /* character contained in one word */
  else {
    mask1 = t->mask << shift1;

    /* display white on black */
    if (mode & INVERSE) {

      /* font bits contained in two bytes */
      if (t->font_width > CHARBITS) {
        for (i = t->font_height - 1; i; i--, b += t->rowints, bits2++)
          *b = LWBCHAR(*b, mask1, *bits2, shift1);
        if (mode & UNDERLINE) *b = LWBCHAR(*b, mask1, t->mask, shift1);
        else *b = LWBCHAR(*b, mask1, *bits2, shift1);
      }

      /* font bits contained in one byte */
      else {
        for (i = t->font_height - 1; i; i--, b += t->rowints, bits1++)
          *b = LWBCHAR(*b, mask1, *bits1, shift1);
        if (mode & UNDERLINE) *b = LWBCHAR(*b, mask1, t->mask, shift1);
        else *b = LWBCHAR(*b, mask1, *bits1, shift1);
      }

    }

    /* display black on white */
    else {

      /* font bits contained in two bytes */
      if (t->font_width > CHARBITS) {
        for (i = t->font_height - 1; i; i--, b += t->rowints, bits2++)
          *b = LBWCHAR(*b, mask1, *bits2, shift1);
        if (mode & UNDERLINE) *b = LBWCHAR(*b, mask1, t->mask, shift1);
        else *b = LBWCHAR(*b, mask1, *bits2, shift1);
      }

      /* font bits contained in one byte */
      else {
        for (i = t->font_height - 1; i; i--, b += t->rowints, bits1++)
          *b = LBWCHAR(*b, mask1, *bits1, shift1);
        if (mode & UNDERLINE) *b = LBWCHAR(*b, mask1, t->mask, shift1);
        else *b = LBWCHAR(*b, mask1, *bits1, shift1);
      }

    }

  }

} /* draw() */

/*****************************************************************************/

/* clear one or more lines */
static void clear(register context_t t,
                  register int x,
                  register int y,
                  register int count)
{
  register unsigned int *b, *bb;

  /* clear a partial line */
  if (x) {
    while (x < t->max_x) space(t, x++, y);
    y++;
    if (--count <= 0) return;
  }

  /* if status line, ignore count */
  if (y == t->max_y) {
    b = &t->buffer[STATUSINDEX(t)];
    for (y = t->font_height; y--; b += t->rowints)
      for (bb = b, x = t->rowsize; x--;) *bb++ = -1;
  }

  /* clear specified number of lines */
  else {
    b = &t->buffer[ROWINDEX(t, y)];

    /* white on black display has no border */
    if (t->white_on_black) {
      while (count--)
        for (y = t->font_height; y--; b += t->rowints)
          for (bb = b, x = t->rowsize; x--;) *bb++ = -1;
    }

    /* right border bits split over two ints */
    else if (t->right2) {
      while (count--) for (y = t->font_height; y--; b += t->rowints) {
        bb = b;
        *bb++ = t->left;
        for (x = t->rowsize - 3; x--;) *bb++ = 0;
        *bb++ = t->right1;
        *bb++ = t->right2;
      }
    }

    /* right border bits fit into one int */
    else {
      while (count--) for (y = t->font_height; y--; b += t->rowints) {
        bb = b;
        *bb++ = t->left;
        for (x = t->rowsize - 2; x--;) *bb++ = 0;
        *bb++ = t->right1;
      }
    }

  }

} /* clear() */

/*****************************************************************************/

/* scroll lines up or down onto specified line */
static void scroll(register context_t t, register int y, int down)
{
  register unsigned int *to_row, *to_int, *from_row, *from_int;
  register int count, x;

  /* scroll from above down onto line */
  if (down) {
    to_row = &t->buffer[ROWINDEX(t, y)];
    from_row = &t->buffer[ROWINDEX(t, y - 1)];
    for (count = (y - 1) * t->font_height; count--;) {
      to_int = to_row;
      from_int = from_row;
      for (x = t->rowsize; x--;) *to_int-- = *from_int--;
      from_row -= t->rowints;
      to_row -= t->rowints;
    }
  }

  /* scroll from below up onto line */
  else {
    to_row = &t->buffer[ROWINDEX(t, y)];
    from_row = &t->buffer[ROWINDEX(t, y + 1)];
    for (count = (t->max_y - y - 1) * t->font_height; count--;) {
      to_int = to_row;
      from_int = from_row;
      for (x = t->rowsize; x--;) *to_int++ = *from_int++;
      from_row += t->rowints;
      to_row += t->rowints;
    }
  }

} /* scroll() */

/*****************************************************************************/

/* blink the display, a visual bell */
static void blink(register context_t t)
{
  register unsigned int *b, *bb;
  register int count, x;

  /* make white black and black white */
  b = &t->buffer[ROWINDEX(t, 0)];
  for (count = t->max_y * t->font_height; count--; b += t->rowints)
    for (bb = b, x = t->rowsize; x--;) *bb = ~*bb++;

  /* make black white and white black */
  b = &t->buffer[ROWINDEX(t, 0)];
  for (count = t->max_y * t->font_height; count--; b += t->rowints)
    for (bb = b, x = t->rowsize; x--;) *bb = ~*bb++;

} /* blink() */

/*****************************************************************************/

/* setup display context */
static void setup(register context_t t,
                  char initial_cursor,
                  int white_on_black)
{
  register unsigned int *b, *bb;
  register int x, y;
  unsigned int filler;

  /* calculate width (in characters) and left margin */
  t->max_x = t->width / t->font_width;
  t->lmargin = (t->width - (t->max_x * t->font_width)) / 2;
  while (t->lmargin < VBORDER) {
    t->max_x--;
    t->lmargin = (t->width - (t->max_x * t->font_width)) / 2;
  }

  /* calculate height (in characters) and top margin */
  t->max_y = t->height / t->font_height;
  t->tmargin = (t->height - (t->max_y * t->font_height)) / 2;
  while (t->tmargin < HBORDER) {
    t->max_y--;
    t->tmargin = (t->height - (t->max_y * t->font_height)) / 2;
  }

  /* status line is y == max_y */
  t->max_y--;

  /* calculate right margin */
  /* this is the first pixel of the right border */
  t->rmargin = t->lmargin + (t->max_x * t->font_width);

  /* calculate rowsize */
  /* this is the number of int's actually displayed */
  t->rowsize = (t->rmargin / INTBITS) + 1;

  /* compute font mask */
  t->mask = (1 << t->font_width) - 1;

  /* make entire display black */
  for (b = t->buffer, y = t->height; y--; b += t->rowints)
    for (bb = b, x = t->rowsize; x--;) *bb++ = -1;

  /* if black on white display, draw borders */
  if (!(t->white_on_black = white_on_black)) {

    /* draw top border */
    border(t, t->tmargin - 3, "11000");
    border(t, t->tmargin - 2, "10011");
    border(t, t->tmargin - 1, "00100");

    /* compute side border bits, clear() draws them */
    bbits(t, "01000", &t->left, &t->right1, &t->right2, &filler);
    clear(t, 0, 0, t->max_y);

    /* draw bottom border */
    border(t, t->max_y * t->font_height + t->tmargin + 0, "00100");
    border(t, t->max_y * t->font_height + t->tmargin + 1, "10011");
    border(t, t->max_y * t->font_height + t->tmargin + 2, "11000");

  }

  /* setup the emulator state */
  t->state = t->mode = t->status = t->status_mode = t->am_count = 0;

  /* draw the initial cursor */
  t->cursor = initial_cursor;
  t->cursor_x = t->cursor_y = 0;
  cursor(t);

} /* setup() */

/*****************************************************************************/

/* setup the font */
/* use internal font if none specified */
/* pick font size if none specified */
static int font(register context_t t,
                char *font_bits,
                int font_height,
                int font_width)
{
  int width;

  /* if height or width not specified, select internal font */
  if (!font_height || !font_width) {
    font_bits = 0;
    width = t->width - VBORDER * 2;
    if ((width / 9) >= 80) {
      font_height = 15;
      font_width = 9;
    }
    else if ((width / 7) >= 80) {
      font_height = 13;
      font_width = 7;
    }
    else {
      font_height = 13;
      font_width = 6;
    }
  }

  /* if font bits not specified, use internal fonts */
  if (!font_bits) {
    if (font_height == 15) {
      if (font_width == 9) font_bits = (char *)font_9x15;
    }
    else if (font_height == 13) {
      if (font_width == 7) font_bits = (char *)font_7x13;
      else if (font_width == 6) font_bits = (char *)font_6x13;
    }
  }

  /* assert that a font has been setup */
  if (!(t->font = font_bits)) return -6;
  if ((t->font_height = font_height) <= 0) return -7;
  if ((t->font_width = font_width) <= 0) return -8;

  /* all done, no error */
  return 0;

} /* font() */

/*****************************************************************************/

/* run terminal emulator to display a character */
void terminal_emulator(terminal_context_t *terminal_context, register char c)
{
  register context_t t = (context_t)terminal_context;

  /* use default context if zero pointer */
  if (!t) t = &default_context;

  /* if margin hit last time, increment the count for possible second time */
  if (t->am_count) t->am_count++;

  /* remove the cursor */
  cursor(t);

  /* a simple state machine */
  switch (t->state) {

    case 0: /* state #0 -- not in escape sequence */
      c &= 127;
      switch (c) {
        case 'A' - '@': /* ^A, enter and clear the status line */
          t->status = 1;
          t->status_mode = 0;
          clear(t, 0, t->max_y, 1);
          break;
        case 'B' - '@': /* ^B, exit the status line */
          t->status = 0;
          break;
        case 'C' - '@': /* ^C, return the termcap entry */
          termcap(t);
          break;
        case 'D' - '@': /* ^D, toggle between white and black */
          setup(t, t->cursor, !t->white_on_black);
          break;
        case 'E' - '@': /* ^E, next character will be new cursor */
          t->state = 5;
          break;
        case 'F' - '@': /* ^F, blink */
          blink(t);
          break;
        case 'G' - '@': /* ^G, beep */
          if (t->bell) (*t->bell)();
          else blink(t);
          break;
        case 'H' - '@': /* le, cursor left */
          if (t->status) {
            if (t->status > 1) t->status--;
          }
          else if (t->cursor_x > 0) t->cursor_x--;
          break;
        case 'I' - '@': /* ^I, tab */
          if (t->status) {
            t->status += (8 - ((t->status - 1) % 8));
            if (t->status > t->max_x) t->status = t->max_x;
          }
          else {
            t->cursor_x += (8 - (t->cursor_x % 8));
            if (t->cursor >= t->max_x) t->cursor_x = t->max_x - 1;
          }
          break;
        case 'J' - '@': /* do, down one line */
          if (t->status) t->status = 0;
          else {
            t->cursor_y++;
            if (t->cursor_y >= t->max_y) {
              t->cursor_y = t->max_y - 1;
              scroll(t, 0, 0);
              clear(t, 0, t->max_y - 1, 1);
            }
          }
          break;
        case 'M' - '@': /* ^M, carriage return */
          if (t->status) t->status = 1;
          else t->cursor_x = 0;
          break;
        case '[' - '@': /* begin escape sequence */
          t->state = 1;
          break;
        default: /* paint the character at cursor position, advance cursor */
          if (t->status) {
            if ((c <= ' ') || (c == 127))
              space(t, t->status++ - 1, t->max_y);
            else draw(t, t->status++ - 1, t->max_y, c);
            if (t->status > t->max_x) t->status = 1;
          }
          else {
            if (t->am_count > 1) { /* am, automatic margin */
              t->cursor_x = 0;
              t->cursor_y++;
              if (t->cursor_y >= t->max_y) {
                t->cursor_y = t->max_y - 1;
                scroll(t, 0, 0);
                clear(t, 0, t->max_y - 1, 1);
              }
              t->am_count = 0;
            }
            if ((c <= ' ') || (c == 127))
              space(t, t->cursor_x++, t->cursor_y);
            else draw(t, t->cursor_x++, t->cursor_y, c);
            /* if at margin, leave the cursor there and start the t->am_count */
            if (t->cursor_x >= t->max_x) {
              t->am_count = 1;
              t->cursor_x--;
            }
          }
      }
      break;

    case 1: /* state #1 -- first char of escape sequence */
      switch (c) {
        case 'Y': /* cm, new cursor position follows */
          t->state = 2;
          break;
        case 'S': /* ts, status line column follows */
          t->state = 4;
          break;
        case 'J': /* cd, clear from cursor to end of display */
          t->state = 0;
          if (t->status) clear(t, t->status - 1, t->max_y, 1);
          else clear(t, 0, 0, t->max_y);
          break;
        case 'K': /* ce, clear to end of line */
          t->state = 0;
          if (t->status) clear(t, t->status - 1, t->max_y, 1);
          else clear(t, t->cursor_x, t->cursor_y, 1);
          break;
        case 'H': /* ho, home cursor */
          t->state = 0;
          if (t->status) t->status = 1;
          else t->cursor_x = t->cursor_y = 0;
          break;
        case 'C': /* nd, cursor right */
          t->state = 0;
          if (t->status) {
            t->status++;
            if (t->status > t->max_x) t->status = t->max_x;
          }
          else {
            t->cursor_x++;
            if (t->cursor_x >= t->max_x) t->cursor_x = t->max_x - 1;
          }
          break;
        case 'I': /* sr, scroll text down */
          t->state = 0;
          if (t->status) break;
          t->cursor_y++;
          if (t->cursor_y >= t->max_y) t->cursor_y = t->max_y - 1;
          scroll(t, t->max_y - 1, -1);
          clear(t, 0, 0, 1);
          break;
        case 'A': /* up, cursor up */
          t->state = 0;
          if (t->status) break;
          t->cursor_y--;
          if (t->cursor_y < 0) t->cursor_y = 0;
          break;
        case 'j': /* so, mr, start standout mode */
          t->state = 0;
          if (t->status) t->status_mode |= INVERSE;
          else t->mode |= INVERSE;
          break;
        case 'k': /* se, end standout mode */
          t->state = 0;
          if (t->status) t->status_mode &= ~INVERSE;
          else t->mode &= ~INVERSE;
          break;
        case 'l': /* us, start underline mode */
          t->state = 0;
          if (t->status) t->status_mode |= UNDERLINE;
          else t->mode |= UNDERLINE;
          break;
        case 'm': /* ue, end underline mode */
          t->state = 0;
          if (t->status) t->status_mode &= ~UNDERLINE;
          else t->mode &= ~UNDERLINE;
          break;
        case 'n': /* as, start alternate (graphic) mode */
          t->state = 0;
          if (t->status) t->status_mode |= GRAPHIC;
          else t->mode |= GRAPHIC;
          break;
        case 'o': /* ae, end alternate (graphic) mode */
          t->state = 0;
          if (t->status) t->status_mode &= ~GRAPHIC;
          else t->mode &= ~GRAPHIC;
          break;
        case 'p': /* md, start bold mode */
          t->state = 0;
          if (t->status) t->status_mode |= BOLD;
          else t->mode |= BOLD;
          break;
        case 'q': /* me, end all modes */
          t->state = 0;
          if (t->status) t->status_mode = 0;
          else t->mode = 0;
          break;
        default:
          blink(t);
      }
      break;

    case 2: /* state #2 -- pick up new cursor Y */
      t->new_y = c - ' ';
      t->state = 3;
      break;

    case 3: /* state #3 -- pick up new cursor X, set new cursor position */
      t->new_x = c - ' ';
      t->state = 0;
      if ((t->new_x < 0) || (t->new_x >= t->max_x)) {
        blink(t);
        break;
      }
      if ((t->new_y < 0) || (t->new_y >= t->max_y)) {
        blink(t);
        break;
      }
      t->cursor_x = t->new_x;
      t->cursor_y = t->new_y;
      break;

    case 4: /* state #4 -- pick up status column */
      t->new_x = c - ' ';
      t->state = 0;
      if ((t->new_x < 0) || (t->new_x >= t->max_x)) {
        blink(t);
        t->new_x = 0;
      }
      t->status = t->new_x + 1;
      break;

    case 5: /* state #5 -- set new cursor character */
      t->state = 0;
      if ((c < 'b') || (c > 'h')) {
        blink(t);
        break;
      }
      t->cursor = TOGRAPHIC(c);
      break;

  } /* switch(t->state) */

  /* restore cursor at new position */
  cursor(t);

  /* if automatic margin ready but not done, clear the count */
  if (t->am_count > 1) t->am_count = 0;

} /* terminal_emulator() */

/*****************************************************************************/

/* switch to a new display buffer */
/* return non-zero if error */
int terminal_buffer(terminal_context_t *terminal_context,
                    char *display_buffer,
                    int display_rowsize)
{
  register context_t t = (context_t)terminal_context;
  register unsigned int *to_row, *to_int, *from_row, *from_int;
  register int x, y, rowints;

  /* use default context if zero pointer */
  if (!t) t = &default_context;

  if (!display_buffer) return -2;

  /* assert that new rows are accessable as (int *) */
  if (display_rowsize % INTBYTES) return -3;

  /* assert that new buffer is (int *) */
  if ((int)display_buffer % INTBYTES) return -4;

  /* assert that new rowsize is sufficient */
  rowints = display_rowsize / INTBYTES;
  if (rowints * INTBITS < t->width) return -5;

  /* if old buffer exists, copy its contents into the new buffer */
  from_row = t->buffer;
  to_row = (unsigned int *)display_buffer;
  if (from_row) for (y = t->height; y--;) {
    to_int = to_row;
    from_int = from_row;
    for (x = t->rowsize; x--;) *to_int++ = *from_int++;
    to_row += rowints;
    from_row += t->rowints;
  }

  /* set new buffer and rowsize */
  t->buffer = (unsigned int *)display_buffer;
  t->rowints = rowints;

  /* all done, no error */
  return 0;

} /* terminal_buffer() */

/*****************************************************************************/

/* prepare a terminal context */
/* return non-zero if error */
int terminal_setup(terminal_context_t *terminal_context,
                   char *display_buffer,
                   int display_height,
                   int display_width,
                   int display_rowsize,
                   char *font_bits,
                   int font_height,
                   int font_width,
                   char initial_cursor,
                   int white_on_black,
                   void (*ungetc)(char, char *),
                   char *ungetc_argument,
                   void (*bell)(void))
{
  register context_t t = (context_t)terminal_context;
  int result;

  /* use default context if zero pointer */
  if (!t) t = &default_context;

  /* set display height and width */
  if ((t->height = display_height) <= 0) return -9;
  if ((t->width = display_width) <= 0) return -10;

  /* set the font */
  result = font(t, font_bits, font_height, font_width);
  if (result) return result;

  /* set display buffer */
  result = terminal_buffer(terminal_context, display_buffer, display_rowsize);
  if (result) return result;

  /* set input write function and its argument */
  t->ungetc = ungetc;
  t->ungetc_argument = ungetc_argument;

  /* set bell function */
  t->bell = bell;
  
  /* setup the rest of the terminal context */
  setup(t, TOGRAPHIC(initial_cursor), white_on_black);

  /* assert that the display is at least 24x80 */
  if (t->max_x < 80) return -11;
  if (t->max_y < 24) return -12;

  /* all done, no error */
  return 0;

} /* terminal_setup() */

/* end of machterm.c */
