/*
 * whom.c -- "who" in columnar format
 *
 * Author:	Steve Scherf
 * Written:	Tue Jun 21 13:14:31 PDT 1988
 *
 * Usage:
 *	whom [-l] [-w screen_width]
 *
 * Options:
 *	-w screen_width - specifies the screen width for wide output
 *	-l		- displays length of login session rather than time
 *
 * Notes:
 *	- Format of output is:
 *		user tty [days[+?]]hours:minutes
 *
 *	  Times of user logins are displayed by default; if the "l" option is
 *	  used, time since login is displayed. In either case, the "days"
 *	  field is always the number of days since login. A "?" after the days
 *	  field indicates that the count has wrapped around some multiple of 10.
 *
 *	- The argument parser is written so that more arguments can easily be
 *	  added.
 *
 *	- Any number of users can be processed.
 */

char *sccsstr = "%W% %E%";

#include <sys/types.h>
#include <sys/utsname.h>
#include <sys/fcntl.h>
#include <sys/stat.h>
#include <stdio.h>
#include <utmp.h>
#include <time.h>

#define  NCOLS		3
#define  SECSPERHOUR	3600
#define  SECSPERDAY	86400

#define  min(x, y)	((x) < (y) ? (x) : (y))

time_t t;
int lflag;
int compar();


main(argc, argv)
int argc;
char **argv;
{
	register int i, j, r, n;
	int ncols = NCOLS;
	int nus, in, nrows, fd;
	char *name;
	struct utsname uts;
	struct stat sbuf;
	struct utmp *user, **ulist;

	name = *argv;

	for(++argv; *argv; argv += 1 + i) {
		if(**argv != '-')
			usage(name);
		i = 0;
		while(*(++*argv))
			switch(**argv) {
				case 'l':
					lflag++;
					break;
				case 'w':
					i++;
					ncols = (atoi(*(argv+i)) + 2) / 27;
					if(ncols < 1)
						usage(name);
					break;
				default:
					usage(name);

				/* plenty of room for more arguments */
			}
	}

	if((fd = open(UTMP_FILE, O_RDONLY)) < 0 || fstat(fd, &sbuf)) {
		perror(UTMP_FILE);
		exit(1);
	}

	if(!(user = (struct utmp *)malloc(sbuf.st_size))) {
		perror("malloc");
		exit(1);
	}

	if(read(fd, user, sbuf.st_size) != sbuf.st_size) {
		perror("read");
		exit(1);
	}
	close(fd);

	for(i = 0, j = sbuf.st_size / sizeof(struct utmp), nus = 0; i < j; i++)
		if(user[i].ut_type == USER_PROCESS)
			nus++;

	if(nus && !(ulist = (struct utmp **)malloc(nus*sizeof(struct utmp)))) {
		perror("malloc");
		exit(1);
	}

	for(i = 0, j = 0; j < nus; i++)
		if(user[i].ut_type == USER_PROCESS)
			ulist[j++] = &user[i];

	qsort((char *)ulist, nus, sizeof(struct utmp *), compar);

	if((t = time(0)) < 0) {
		perror("time");
		exit(1);
	}

	uname(&uts);
	printf("%d user%s on %s @ %s", nus, (nus == 1 ? "" : "s"),
	    uts.nodename, ctime(&t));

	r = nus / ncols;
	n = nus % ncols;
	nrows = r + (n ? 1 : 0);

	for(i = 0; i < nrows; i++) {
		for(j = 0; j < ncols; j++)
			if((in = j * r + i + min(j, n)) < nus &&
			    !(n && j >= n && i == nrows - 1)) {
				if(j != 0)
					printf("   ");
				printu(ulist[in]);
			}
		putchar('\n');
	}
}


int
compar(u1, u2) 
struct utmp **u1, **u2;
{
	register int val;

	if(!(val = strcmp((*u1)->ut_name, (*u2)->ut_name)))
		val = strcmp((*u1)->ut_line, (*u2)->ut_line);

	return val;
}


printu(u)
struct utmp *u;
{
	register unsigned int hr, mn, sec, dazon;
	struct tm *tm;
	char buf1[10], buf2[10];

	buf1[0] = '\0';
	if(dazon = (unsigned int)(t - u->ut_time) / SECSPERDAY)
		sprintf(buf1, "%d%c", (dazon % 10), (dazon > 9 ? '?' : '+'));

	if(lflag) {
		sec = (unsigned int)(t - u->ut_time) % SECSPERDAY;
		hr = sec / SECSPERHOUR;
		mn = (sec % SECSPERHOUR) / 60;
	}
	else {
		tm = localtime(&u->ut_time);
		hr = tm->tm_hour;
		mn = tm->tm_min;
	}
	sprintf(buf2, "%.2d:%.2d", hr, mn);

	u->ut_line[7] = '\0';
	u->ut_name[8] = '\0';

	printf("%-8s %-7s %2s%-5s", u->ut_name, u->ut_line, buf1, buf2);
}


usage(name)
char *name;
{
	fprintf(stderr, "usage: %s [-l] [-w screen_width]\n", name);
	exit(1);
}
