/*************************************************************************
*
*
*	Name:  errs.c
*
*	Description:  Error handler & message diplay routines.
*					sfmsg()			- Show a message on the error line.
*					sfbeep()		- Beep.
*					sfnotimp()		- Report unimplemented feature.
*					sfsuecod()		- Show error msg, using user error code.
*					sfclrmsg()		- Clear error message line.
*					sf_errmsg()		- Show message on error line.
*					sf_erwmsg()		- Show error msg & wait for CLRERR.
*					sf_errwait() 	- Wait for 'Clear Error' fn key.
*					sf_panic()		- Panic.
*
*
*	History:
*	Date		By		Comments
*
*	03/14/84	waf
*	05/10/84	waf		Call sfexit() after panic.
*	06/08/84	waf		**  Rev. 2.0  **
*
*
*
*  Copyright (c) 1983, 1984 by Digital Communication Assoc..
*  This document contains confidential/proprietary information.
*
*************************************************************************
*  SForm routines module.  */




/*
  Notes -

  All error output is made to stdout (not stderr).

*/

#include	"/sform/src/sfint.h"

extern int	sfmsgflg ;				/* Set after message displayed */



sfmsg ( msg )
char	*msg ;
/*
  Show a error/status message on the error line.
  The sfmsgflg flag is set, so that message will be deleted later.
*/
{

	ermset() ;
	stputf(SF_ERRROW, 0, SF_MAXCOL, SF_NORM, msg) ;
	sfmsgflg = 1 ;
	ermrst() ;
}



sfclrmsg ()
/*
  Clear the error message line & reset the sfmsgflg flag.
  NOTE - the cursor position is not saved.
*/
{

	stclw(SF_ERRROW, SF_ERRROW, 0, SF_MAXCOL, SF_NORM) ;
	sfmsgflg = 0 ;
}



sf_errmsg ( msg )
char	msg[] ;
/*
  Report error on error line.
  Note - The 'sferrwait()' fn can not be called, because there may be
  a user error handler. If the user does not trap the error, the sferrwait()
  fn should be invoked by the caller.
*/
{

	errmsg(msg, 0) ;
}



sf_erwmsg ( msg )
char	msg[] ;
/*
  Report an error and wait for 'Clear Error' fn key.
  This fn should be used whenever the error can not be trapped by the
  user.
*/
{

	errmsg(msg, 1) ;
}



static	errmsg ( msg, waitflag )
char	msg[] ;
int		waitflag ;
/*
  Show error msg.
  If waitflag is set, wait for 'Clear Error' fn key, then clear msg.
*/
{

	sfbeep() ;
	sfmsg(msg) ;
	if (waitflag) {
		sf_errwait() ;
		sfclrmsg() ;
	}
}



sf_errwait ()
/*
  Wait for 'Clear Error' or 'Help' fn key to be hit.
  Beep if any other key is hit.
*/
{
	register int	c ;

	while ((c=stgetc()) != SF_CLRERR) {
		if (c == SF_HELP) {
			(*sf_usrhelp)() ;
			break ;
		}
		sfbeep() ;
	}
}



sfbeep ()
/*
  Output a 'beep'.
*/
{

	stbeep(SF_BPNORM) ;
}

sf_panic ( msg )
char	*msg ;
/*
  Report unmanagable error and exit.
  Note - message should include fn name (for debugging).
*/
{

	stputs("\7**>> SForm panic - ") ;
	stputs(msg) ;
	sfterm() ;			/* return terminal control */
	stputc('\n') ;
	exit(-1) ;
}



sfnotimp ()
/*
  We tried to invoke a feature/function which is not yet implemented.
  Show message and continue.
  Note - cursor position not saved.
*/
{

	sf_erwmsg("Feature is not currently implemented.") ;
}

static	ermset ()
/*
  Get ready to print error message.
*/
{

	/** Nothing to do **/
}



static	ermrst ()
/*
  Do houskeeping after error message display.
  The cursor is repositioned to the 1st char in the current fld.
  The global var 'sf_curfd' is used as the fld desc for the current fld.
*/
{
	register int	row, col ;
	register struct SF_FIELD *fld_desc ;

	if (sf_curwin != SF_NULL) {		/* if a window is open */
		fld_desc = sf_curfd ;		/* current fld */
		row = fld_desc->sf_row ;	/* current row */
		col = fld_desc->sf_col ;	/* start col of current fld */
		stscp(row, col) ;			/* set cursor pos */
	}
}

sfsuecod ( errcode )
int		errcode ;
/*
  Show error message coresponding to error code.
  This is the default action taken when a user error occurs.
*/
{
	char		*msg ;
	extern int	sfmioflg ;

	switch ( errcode ) {
		case SF_ENOVF :
			msg = "Numeric overflow." ;
			break ;
		case SF_ENCNV :
			msg = "Numeric conversion error." ;
			break ;
		case SF_ENRNG :
			msg = "Numeric value out of range." ;
			break ;
		case SF_EENMS :
			msg = "Illegal value." ;
			break ;
		case SF_ESCHR :
			msg = "Illegal character(s) in input data." ;
			break ;
		case SF_ENUNQ :
			msg = "Abbreviation must be unique." ;
			break ;
		case SF_EINVV :
			msg = "Invalid value." ;
			break ;
		default :
			msg = "<* Unknown error code *>" ;
	}

	if (sfmioflg)
		/* Don't show error msg's during sfmsgio() */
		sfbeep() ;
	else
		/* Show error msg */
		sf_erwmsg(msg) ;

	return(SF_OK) ;			/* this MUST be returned */
}
