/*************************************************************************
*
*
*	Name:  bberr.c
*
*	Description:  Error handler fn's.
*
*			bberr()	- does a long jump back to the setjmp call in pmach.
*			bbxerrno() - translate error code in 'errno', then do bberr().
*			panic()	- does an uncapturable error 4. This is used to stop
*					the p-machine in the case of an error that shouldn't happen.
*			ioerr()	- processes i/o errors and EINTR errors.
*
*
*	History:
*	Date		By	Comments
*
*	03/01/83	mas
*	04/13/83	mas	error numbers <= zero are shifted down
*				by -1 to get through the longjmp/setjmp
*				check in pmach.c. The zero returned from
*				setjmp signifies the setting of the long
*				jmup vector and therefore cannot be used
*				to pass an error number of zero.
*	10/05/83	waf		Added ioerr().
*	11/10/83	waf		panic() - save ret addr of caller in 'panicpc'.
*	11/16/83	waf		Added bbxerrno().
*	02/21/84	waf		Change functionality of ioerr(). See note below.
*
*
*
*  This document contains confidential/proprietary information.
*
*  Copyright (c) 1983, 1984 by Digital Communications Assoc.
*
*************************************************************************
* BB/Xenix Runtime Module */



/*  Notes -



Modification Notes -

11/7/83		waf
  A panic() can be generated by p-machine code (when strangeness
occurs during execution), or by the ABORT p-code (output by the compiler
to handle strangeness).
  When a panic() occurs, the ret addr of the calling code is saved in
'panicpc'. This value is used later to dump the pc as part of the error msg.

02/21/84	waf
  The method of handling EINTR errors was changed. Previously, the state of
the 'ust.noikey' flag was returned to the caller if the error code was EINTR.
This was changed so that, if noikey is set, the error is reported. Thus,
ioerr() will return only if the error is EINTR and the noikey flag is reset.
In this case, the ikey will be acknowleged at the next STMTX.
  NOTE that ikey'ed char i/o ABORTS the i/o (in an UNKNOWN state) and returns
an EINTR error. The calling code must be set up to handle a return from 
ioerr() (e.g. when ikey's are enabled).

*/

#include "/bb/include/ptype.h"
#include "/bb/include/pextern.h"
#include "/bb/include/bberms.h"
#include <errno.h>



bberr(bberrno)

/* Handle error.
*/

int	bberrno;	/* BB error code */
{

	if (bberrno <=0)
		--bberrno;	/* adjust zero error number */

	longjmp(errenv,bberrno);
}



bbxerrno()

/* Translate system error code in 'errno', and report bb error.
  (Saves 14 bytes over 'bberr(errxlt(errno))'.
*/
{

	bberr( errxlt(errno) );
	}

#define		RAOS	(-1)	/* ret addr (int) offset */

int		panicpc;			/* save ret addr here */


panic (dummy)

int	dummy;	/* dummy var used to get caller's addr. */

/*
  Panic error - uncapturable by Basic user.
  Save ret addr of caller in 'panicpc', then abort (via bberr()).

  NOTE - assumes ret addr is sizeof(int), and RAOS is correct offset.
*/

{
	register int	*ptr;


	/* get caller's ret addr */
	ptr = (int *)&dummy ;
	panicpc = *(ptr + RAOS);		/* caller's ret addr */

	longjmp(errenv,BESYS);
}



ioerr ()

/*
  Synopsis -
	Process an i/o call error.
	If the error is not EINTR, process the error normally.
	If the error is EINTR, chk state of noikey flag.

  Description -
	  (When an ikey occurs during character i/o, the i/o system call is 
	immediately aborted, and an EINTR error code is returned in 'errno'.
	Selected i/o calls in the p-mach will call ioerr() after an error is
	detected, in order to check for the EINTR error, and process accordingly.)

	  This fn is called after an i/o call (read()/write()) has returned an
	error status, and all local processing of the error condition is finished.
	The global var 'errno' is assumed to contain the error code.
	The action taken by this function is determined by the value of the error
	code and the state of the 'noikey' flag.
	  If the error code is not EINTR, the error is processed normally (it is
	assumed that a 'normal' i/o error ocurred and should be reported).
	Note that the calling code must have already chk'ed for any error codes
	which could be handled locally, such as end of file, before calling this
	fn. Calling this fn with a 'normal' (non EINTR) i/o error is functionally
	the same as calling 'bbxerrno()' directly.
	  If the error code is EINTR, the i/o call must have been to a character
	oriented device, and an interrupt must have occured while the i/o was
	taking place (this is an artifact of Xenix i/o to character devices).
	In this case, the noikey flag is used to determine what to do.
	If noikey is set (via STMA 6,5), then the error MUST be reported
	immediately, because ikey'ed char i/o is aborted in an unknown state
	(there is not enough info returned for the i/o call to be resubmitted,
	thus the ikey CAN NOT BE IGNORED).
	If the noikey flag is rest (e.g. ikeys are 'enabled'), then an EINTR
	error is IGNORED, because the ikey will be processed by the next STMT
	p-code. In this case, control is returned to the caller, which should
	have code to abort the rest of the Basic statement.

  Return -
	Only returns if the error was EINTR (e.g. ikey'ed char i/o), and the
	noikey flag is reset. In this case, the caller should have code to
	'abort' the rest of the Basic statement.


	** NOTE - when i/o to a char device is interrupted, the state of the i/o
  transfer is UNDEFINED. This situation may cause garbage to be stored in
  a BB variable if data is input directly to the variable's data space.

    Note that an 'interrupt' is handled like an 'error' during char i/o.
  This creates confusion to the user (and the programmer).

	Note that the calling code should be able to handle a return from the
  ioerr() fn. Generally, this code would skip the rest of the BB statement.
  For example, if an ikey occurs during an input stmt, and there are multiple
  input vars, the assignment to the vars should be aborted as soon as the ikey
  is detected (since the state of the input buffer is undefined).

    Note that it is assumed that the 'longjmp()' in the onikey.c code is
  NOT taken. This is normally the case, as the default value of 'valikey'
  is 'FALSE'. If the longjmp() is taken, then the i/o call would never
  return, and the call to ioerr() would never occur.
*/

{


	if ( errno != EINTR )


		/** i/o error (not ikey) - process the error **/
		bbxerrno() ;


	else {


		/** interrupted char i/o - chk the noikey flag **/
		if ( ust.noikey )
			bbxerrno();		/* ikeys diabled - report error now */
		else
			return ;		/* ikeys enabled - ignore ikey,
								it will be handled at next STMTX */
		}

	}
