/*************************************************************************
*
*
*	Name:  genio.c
*
*	Description:  Perform generic i/o on a field.
*
*
*	History:
*	Date		By		Comments
*
*	06/08/84	waf		**  Rev. 2.0  **
*	07/02/84	waf		NOECHO support.
*
*
*
*  Copyright (c) 1983, 1984 by Digital Communication Assoc..
*  This document contains confidential/proprietary information.
*
*************************************************************************
*  SForm routines module.  */




/*  Notes -
	
  This is a low level routine, invoked by the type specific routines.
  The data has been converted into a string fld before this fn is invoked.

  NOTE that trailing spaces are stripped on entry, and leading spaces/zeroes
  and trailing spaces are stripped on exit from this routine.

*/

#include	"/sform/src/sfint.h"



sfgenio ( fld_desc, mode, fld_data )
struct SF_FIELD	*fld_desc ;
int				mode ;
char			*fld_data ;			/* string data */
/*
  Synopsis -
	Do i/o on a string field.

  Description -
	This fn does the low level flow control necessary to display or edit
	a string field. If the user enters new data in the field, the data
	is returned to the calling function for conversion & validation.
	The field data has been converted to string, if necessary, by the calling
	fn.
	This fn handles display justification. Before display, the fld is filled
	out with justification fill chars. If the fld is edited, the justification
	fill chars are stripped before returning the new data to the caller.

  Return -
	return val	= SForm completion code.

  Notes -
  > Note that st_mdt is not reset at this level. The calling code MUST RESET
  st_mdt before the initial call to sfgenio().
  > Currently, a second scratch data area is being used to expedite the
  justification & stripping of the string data. This may not be practical
  if dynamic allocation were used.
  > NOECHO flds are displayed as all blanks.
*/
{
	int				r, c, fwid ;
	unsigned		attr ;
	int				sfc ;
	char			*tmp_data ;
	char			*sfjust() ;
	int				flags ;
	extern int		sfmsgflg ;


	/* Get fld params */
	r = fld_desc->sf_row ;
	c = fld_desc->sf_col ;
	attr = fld_desc->sf_iodata->sf_datr ;
	fwid = fld_desc->sf_width ;

	/* Get flags */
	flags = fld_desc->sf_iodata->sf_flags ;

	/* Strip any trailing spaces */
	spstrip(fld_data, fwid) ;

	/* Create a justified fld string */
	tmp_data = sfjust(fld_desc, fld_data) ;

again:
	/* Display the fld and (optionally) get user input */
	sfc = stgetf(r, c, fwid, attr, tmp_data, mode, flags) ;

	/* Clear error/status msg */
	if (sfmsgflg)
		sfclrmsg() ;

	/* Chk comp code */
	switch (sfc) {
		case  SF_CLRERR :
			sfbeep() ;				/* invalid unpend */
			goto again ;
	}

	/* 'Un-justify' the fld (strip leading/trailing blanks),
	   and copy input data back into fld_data */
	if (st_mdt)
		sfunjust(fld_desc, tmp_data, fld_data) ;
	

	/* Return comp code */
	return(sfc) ;
}

#if	(SF_BUFSIZ != -1)
static	char	justbuf[SF_BUFSIZ] ;
#endif



static	char	*sfjust ( fld_desc, fld_data )
struct SF_FIELD	*fld_desc ;
char			*fld_data ;
/*
  Synopsis -
	Justify the fld data string, using justification info from fld_desc.

  Description -
	fld_data contains a string which is the string representation of the fld.
	Justification/zero fill is done on the string at this level. Before
	returning, any justification fill chars are stripped from the front and
	back of the string.

  Return -
	return val	=	Ptr to justified string.

  Notes -
  *> The input string is assumed left justified and null terminated.
	 The (possibly modified) string is returned the same way.
  *> Currently, a second scratch buffer is used. This makes justification
	 easier and faster.
  > Note that 'justification' is done at this level by adding blanks/zeros.
  > Zero fill implies right justification.
*/
{
	register char	*cp ;
	register int	rs, ls ;
	int				ws ;
	int				fwid ;
	int				flags ;
	int				sl ;
	char			c ;
	char			*jstr ;

	/* Get bit flags */
	flags = fld_desc->sf_iodata->sf_flags ;

	/* Get fld width */
	fwid = fld_desc->sf_width ;

	/* Get some scratch memory for justified string */
#if	(SF_BUFSIZ != -1)
	jstr = justbuf ;
#else
	jstr = sfalloc(fwid) ;
#endif
	cp = jstr ;

	/* Get # spaces on left & right of str */
	ls = rs = 0 ;				/* assume none */
	sl = strlen(fld_data) ;
	if (sl > fwid)
		sl = fwid ;				/* fld overflow - truncate str */
	if (sl < fwid) {
		ws = fwid - sl ;		/* white space in fld */
		if (flags & SF_CJUST) {
			/* Center justified */
			ls = ws / 2 ;
			rs = ws - ls ;
		}
		else if ((flags & SF_RJUST) || (flags & SF_ZFILL)) {
			/* Right justified */
			ls = ws ;
		}
		else {
			/* Left justified */
			rs = ws ;
		}
	}
	
	/* Insert left side spaces/zeroes */
	if (flags & SF_ZFILL) {
		c = '0' ;
		/* Chk for minus sign.
		   If exists, show '-' before leading zeroes. */
		if (*fld_data == '-') {
			*cp++ = '-' ;
			fld_data++ ;
		}
	} else
		c = ' ' ;
	while (ls--)
		*cp++ = c ;

	/* Insert string */
	while (c = *fld_data++)
		*cp++ = c ;

	/* Insert right side spaces */
	while (rs--)
		*cp++ = ' ' ;
	
	/* Term str */
	*cp = '\0' ;

	/* Return ptr to justified string */
	return(jstr) ;
}



static	sfunjust ( fld_desc, instr, outstr )
struct SF_FIELD	*fld_desc ;
char			*instr, *outstr ;
/*
  'Unjustify' the string.
  The instr is copied to the outstr, stripping any leading blanks/zeroes and
  trailing blanks.
  Note that, if SF_ZFILL is set, leading zeroes are stripped, else leading
  blanks are stripped.
*/
{
	register char	*src, *dst ;
	char			lc ;

	src = instr ;
	dst = outstr ;

	/* Get leading char */
	if (fld_desc->sf_iodata->sf_flags & SF_ZFILL)
		lc = '0' ;
	else
		lc = ' ' ;
	
	/* Skip leading blanks/zeroes */
	while (*src == lc)
		src++ ;
	
	/* Copy the string data, including null */
	while (*dst++ = *src++)
		;

	/* Strip trailing blanks */
	dst-- ; dst-- ;					/* last char */
	while (*dst == ' ')
		*dst-- = '\0' ;
}

static	spstrip ( str, fwid )
char	*str ;
int		fwid ;
/*
  Strip trailing spaces from the string fld.
  Upon exit, the string is null termed, with no trailing spaces.
  Note - if the string is all spaces, it is truncated to a null string.
*/
{
	register char	*cp, *ep ;

	ep = str - 1 ;				/* go to the beginning */
	cp = ep + fwid ;			/* -> last char */
	while ((cp > ep) && (*cp == ' '))
		*cp-- = '\0' ;
}
