/* memmap.h -- RAM-Monitor mapping procedure macros
 *	(c)
 *	Daniel Steinberg
 *	March, 1984
 */
#ifndef MEMMAP
#define MEMMAP

/* All references (hopefully) to the program memory space of the target task
 * being debugged use the routines in memory.S to read and write arbitrary
 * locations.  Such routines depend on a flag that specifies whether the
 * memory reference should be taken as a physical address (PHYS), a mapped
 * supervisor address (T_SUPER), or a mapped user address (T_USER).  This flag
 * is either hard-coded (e.g., transparent breakpoint setting), forced to PHYS
 * (by setting CF_PHYS ( "p e" command), or set by getmap() (in rammonitor.c)
 * to reflect the current running state of the target task  (which involves
 * decoding PSR_U, MSR_TS, MSR_TU, and MSR_DS).
 *
 * Routines that access memory in a variety of ways have a number of pseudonyms
 * defined below.  The general strategy is as follows:
 *	1) routine(...)  calls mroutine(..., map) where map is PHYS (if CF_PHYS
 *			 is set) or the current mapping from getmap()
 *	2) proutine(...) calls mroutine(..., PHYS)
 *	3) vroutine(...) calls mroutine(..., getmap()) to override CF_PHYS
 *	4) mroutine(..., map) is the actual routine; 'map' is passed to the
 *			 memory access routines
 */

#define PHYS	1
#define T_SUPER	2
#define T_USER	3

#define Nmap ((cflags & CF_PHYS) ? PHYS : getmap())

#define setbrk(addr,name,type)  msetbrk(addr,name,type, Nmap)
#define psetbrk(addr,name,type) msetbrk(addr,name,type, PHYS)
#define vsetbrk(addr,name,type) msetbrk(addr,name,type, getmap())

#define isbrkset(addr,type)  misbrkset(addr,type, Nmap)
#define pisbrkset(addr,type) misbrkset(addr,type, PHYS)
#define visbrkset(addr,type) misbrkset(addr,type, getmap())

#define disasm(a,b)  mdisasm(a,b, Nmap)
#define pdisasm(a,b) mdisasm(a,b, PHYS)
#define vdisasm(a,b) mdisasm(a,b, getmap())

#define membyte(addr)  mrdbyte(addr, Nmap)
#define pmembyte(addr) mrdbyte(addr, PHYS)
#define vmembyte(addr) mrdbyte(addr, getmap())

#define memword(addr)  mrdword(addr, Nmap)
#define pmemword(addr) mrdword(addr, PHYS)
#define vmemword(addr) mrdword(addr, getmap())

#define memquad(addr)  mrdquad(addr, Nmap)
#define pmemquad(addr) mrdquad(addr, PHYS)
#define vmemquad(addr) mrdquad(addr, getmap())

#define wrtmbyte(addr, val)  mwrtmbyte(addr, Nmap, val)
#define pwrtmbyte(addr, val) mwrtmbyte(addr, PHYS, val)
#define vwrtmbyte(addr, val) mwrtmbyte(addr, getmap(), val)

#define wrtmword(addr, val)  mwrtmword(addr, Nmap, val)
#define pwrtmword(addr, val) mwrtmword(addr, PHYS, val)
#define vwrtmword(addr, val) mwrtmword(addr, getmap(), val)

#define wrtmquad(addr, val)  mwrtmquad(addr, Nmap, val)
#define pwrtmquad(addr, val) mwrtmquad(addr, PHYS, val)
#define vwrtmquad(addr, val) mwrtmquad(addr, getmap(), val)

#endif
